# Copyright (C) 2020 INAF
# This software is distributed under the terms of the BSD-3-Clause license
#
# Authors:
#    Bulgarelli Andrea <andrea.bulgarelli@inaf.it>
#    Baroncelli Leonardo <leonardo.baroncelli@inaf.it>
#    Parmiggiani Nicolò <nicolo.parmiggiani@inaf.it>
#    Addis Antonio <antonio.addis@inaf.it>

from time import sleep

import SAG_MODULE
import SAG_MODULE__POA

import SAG_CUSTOM_TYPES_MODULE

from Acspy.Servants.ACSComponent import ACSComponent
from Acspy.Servants.ComponentLifecycle import ComponentLifecycle
from Acspy.Servants.ContainerServices import ContainerServices

class SAGSubArrayMonitorImpl(SAG_MODULE__POA.SAGSubArrayMonitor, ACSComponent, ComponentLifecycle, ContainerServices):
    
    """
    This interface exposes methods to start/stop the internal monitoring of the SAG. It is used internally within the SAG system.    
    """
    
    def __init__(self):
        ACSComponent.__init__(self)
        ContainerServices.__init__(self)
        
    def initialize(self):
        self.monitor = False
        self._logger = self.getLogger()

    def execute(self):
        self.notificationManager = self.getComponent("PY_SAGNOTIFICATION_MANAGER")
        
    def cleanUp(self):
        self.stopMonitoring()
        self.releaseComponent("PY_SAGNOTIFICATION_MANAGER")
        
    def aboutToAbort(self):
        #Do any critical clean up
        #Continue with less critical stuff such as releasing components and other activities similar to cleanUp
        pass
 
    def getSubArraysID(self, componentsNames):
        return set([compName.split("_MANAGERID_")[1] for compName in componentsNames])
        
    def getComponentsOfSubArray(self, components, subarrayID):
        return [comp for comp in components if "MANAGERID_"+subarrayID in comp.name]

    def startMonitoring(self):

        """
		[Async] Start the monitoring of the SAGReco, SAGDataQuality and SAGSci components for each scheduling block. The monitoring data is sent to external subsystems.
        """
    
        self._logger.logInfo("[SAGSubArrayMonitorImpl - startMonitoring] Start monitoring")

        self.monitor = True

        while self.monitor:
            
            self._logger.logInfo("[SAGSubArrayMonitorImpl - startMonitoring] monitoring..")
            
            sleep(2)

            components = self.availableComponents(name_wildcard="SAG_PROCESS_*", activated=1) #SAG_PROCESS
            
            if len(components) == 0:
                self._logger.logInfo("[SAGSubArrayMonitorImpl - startMonitoring] no SAG_PROCESS_* components found..")
                continue

            componentsNames = [comp.name for comp in components]
            self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] Components: {componentsNames}")

            subarrayManagersIds = self.getSubArraysID(componentsNames)
            self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] subarrayManagersIds: {subarrayManagersIds}")

            
            for subarrayID in subarrayManagersIds:

                subarrayComponents = self.getComponentsOfSubArray(components, subarrayID)
                self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] subarrayComponents for {subarrayID}: {subarrayComponents}")


                # Sending event to MON 
                eventToMon = SAG_CUSTOM_TYPES_MODULE.SAGChannelEvent_TO_MON(0,0,0,0,0,0,0,0,0)

                for component in subarrayComponents: # reco/dq/sci
        
                    self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] Asking component: {component.name}")

                    component = self.getComponent(component.name)

                    monitoredParamsList = component.getMonitoringInfo()

                    monitoredParamsDict = {}

                    for monObj in monitoredParamsList: monitoredParamsDict[monObj.name] = monObj.value                
                    
                    # Save monitoring info in database?

                    # building message for MON
                    if "RECO" in component.name:    
                        self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] component {component.name} getMonitoringInfo: {monitoredParamsDict}")                    
                        eventToMon.timestamp = int(monitoredParamsDict["timestamp"])
                        eventToMon.schedulingBlockID = int(monitoredParamsDict["schedulingBlockID"])
                        eventToMon.inputDataRate = float(monitoredParamsDict["inputDataRate"])
                        eventToMon.inputEventRate = float(monitoredParamsDict["inputEventRate"])
                        eventToMon.recoStepOneMeanProcessingRate = float(monitoredParamsDict["recoStepOneMeanProcessingRate"])
                        eventToMon.recoStepOneMeanProcessingRate = float(monitoredParamsDict["recoStepOneMeanProcessingRate"])
                        self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] RECO updating message to MON: {eventToMon}")
                    
                    elif "DQ" in component.name:
                        self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] component {component.name} getMonitoringInfo: {monitoredParamsDict}")                    
                        eventToMon.dqStepOneMeanProcessingRate = float(monitoredParamsDict["dqStepOneMeanProcessingRate"])
                        eventToMon.dqStepTwoMeanProcessingRate = float(monitoredParamsDict["dqStepTwoMeanProcessingRate"])
                        self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] DQ updating message to MON: {eventToMon}")
                    
                    elif "SCI" in component.name:
                        self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] component {component.name} getMonitoringInfo: {monitoredParamsDict}")                    
                        eventToMon.blindSearchDetectionAnalysisMeanRate = float(monitoredParamsDict["blindSearchDetectionAnalysisMeanRate"])
                        self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] SCI updating message to MON: {eventToMon}")



                self._logger.logInfo(f"[SAGSubArrayMonitorImpl - startMonitoring] event to MON : {eventToMon}")

                if self.notificationManager:

                    self.notificationManager.sendMessageToMON(eventToMon)


    def stopMonitoring(self):
        """
        Stop the monitoring of the SAGReco, SAGDataQuality and SAGSci components.
        """
        self._logger.logInfo(f"[SAGSubArrayMonitorImpl - stopMonitoring] self.monitor is {self.monitor} setting it to False")
        if self.monitor:
            self.monitor = False
            return True
        else:
            return False


    def addRequester(self, subsystemName):
        """
        [TBI] Allow an external subsystem to receive the monitoring data. 
        """
        self._logger.logInfo(f"[SAGSubArrayMonitorImpl - addRequester] Adding {subsystemName}")
        return True

    def removeRequester(self, subsystemName):
        """
        [TBI] Prevent an external subsystem to receive the monitoring data. 
        """
        self._logger.logInfo(f"[SAGSubArrayMonitorImpl - removeRequester] Removing {subsystemName}")
        return True