import os
import pytest
import pathlib
import subprocess
from time import sleep
from Acspy.Clients.SimpleClient import PySimpleClient



@pytest.fixture(scope="function")#, autouse=True)
def acsEnvironment(request):
    """
    It starts the acs framework with a subprocess call. The acs framework is stopped after the testing session.
    Raises:
        ValueError: if the ACS_CDB environment variable is not defined
    Returns:
        The return code of the acsStart command
    """
    print("[acsEnvironment fixture] Starting acs..")

    marker = request.node.get_closest_marker("test_name")
    if marker is None:
        raise ValueError("marker is None! Something wrong passing data to fixture!")
    test_name = marker.args[0]

    ACS_CDB = os.environ.get("ACS_CDB")
    if ACS_CDB is None:
        raise ValueError("ACS_CDB environment variable is not defined.")
    logFolder = pathlib.Path(__file__).parent.absolute().joinpath("test_logs/acs").joinpath(test_name)
    logFolder.mkdir(exist_ok=True, parents=True)    
    logFile = open(str(logFolder.joinpath("acsStart.log")), "w")
    returncode = subprocess.call("acsStart", stdout=logFile, stderr=logFile)
    print(f"[acsEnvironment pytest fixture] acsStart command return code: {returncode}")
    yield returncode
    logFile.close()
    print("[acsEnvironment pytest fixture] Stopping acs")
    subprocess.call(["acsStop"])   
    print("[acsEnvironment pytest fixture] Killing acs with KillACS")
    subprocess.call(["killACS", "-q", "-a"])


@pytest.fixture(scope="function")
def acsContainers(request):
    """
    It starts acs containers. The acs containers stopped at the end of the tests when acs is shutted down.
    Args:
        containers_name (str): the name of the containers separated by a comma
    Raises:
        ValueError: if the marker object is None
    Returns:
        The pid of the container's process (int)
    """
    print("[acsContainers fixture] Starting containers..")

    marker = request.node.get_closest_marker("test_name")
    if marker is None:
        raise ValueError("marker is None! Something wrong passing data to fixture!")
    test_name = marker.args[0]

    marker = request.node.get_closest_marker("containers_name")
    if marker is None:
        raise ValueError("marker is None! Something wrong passing data to fixture!")
    containers_name = marker.args[0]

    if "," in containers_name:
        containerNames = containers_name.split(",")
    logFolder = pathlib.Path(__file__).parent.absolute().joinpath("test_logs/acs").joinpath(test_name)
    logFolder.mkdir(exist_ok=True, parents=True)
    pids = []
    logFiles = []
    for containerName in containerNames:
        logFile = open(str(logFolder.joinpath(containerName)), "w")
        logFiles.append(logFile)
        process = subprocess.Popen(["acsStartContainer", "-py", containerName], stdout=logFile, stderr=logFile)
        pids.append(process.pid)
    sleep(10)
    print("[acsContainers fixture] Containers started!!")
    yield pids
    print("[acsContainers fixture] Stopping containers..")
    for idx, containerName in enumerate(containerNames):
        process = subprocess.Popen(["acsStopContainer", containerName], stdout=logFiles[idx], stderr=logFiles[idx])
    for logFile in logFiles:
        logFile.close()
    sleep(5)
    print("[acsContainers fixture] Containers stopped!!")


@pytest.fixture(scope="function")
def acsPyClient():
    """
    It creates a PySimpleClient object. The PySimpleClient object is destroyed after the testing class is destroyed.
    Returns:
        The PySimpleClient object
    """
    print("[acsPyClient fixture] acquiring client..")
    client = PySimpleClient()
    yield client
    print("[acsPyClient fixture] disconnecting client..")
    client.disconnect()

@pytest.fixture(scope="function")
def cleanSimulatedProcesses(request):
    print("[cleanSimulatedProcesses fixture] Killing sim_pipe_* processes...")

    marker = request.node.get_closest_marker("test_name")
    if marker is None:
        raise ValueError("marker is None! Something wrong passing data to fixture!")
    test_name = marker.args[0]

    logFolder = pathlib.Path(__file__).parent.absolute().joinpath("test_logs/acs").joinpath(test_name)
    logFolder.mkdir(exist_ok=True, parents=True)    
    logFile = open(str(logFolder.joinpath("pkill.log")), "w")
    returncode = subprocess.call(["pkill", "-f", "sim_pipe"], stdout=logFile, stderr=logFile)
    print(f"[cleanSimulatedProcesses fixture] pkill return code: {returncode}")
    logFile.close()
    return returncode

@pytest.fixture(scope="function")
def printTestName(request):
    marker = request.node.get_closest_marker("test_name")
    if marker is None:
        raise ValueError("marker is None! Something wrong passing data to fixture!")
    test_name = marker.args[0]
    print(f"\n****************** [{test_name}] ****************** ")
    return test_name
