import configparser
from pathlib import Path

import numpy as np
from astropy.coordinates import SkyCoord, EarthLocation
from astropy.coordinates import get_body, solar_system_ephemeris
from astropy.coordinates.name_resolve import NameResolveError
from astropy.time import Time
from astropy.io import fits
from astropy import units as u
from astropy import log

class Noche:
    """
    This class provides tools to initialize and populate a FITS header for
    imaging or spectroscopic observations based on a configurable header
    template and site-specific observatory data.
    
    Parameters
    ----------
    header_template_path : str, optional
        Path to the .ini configuration file with base header structure.
    
    Attributes
    ----------
    header : astropy.io.fits.Header
        The FITS header object being constructed.
    _coord : astropy.coordinates.SkyCoord or None
        Sky coordinates of the observed object.
    _obstime : astropy.time.Time or None
        Observation time.
    _location : astropy.coordinates.EarthLocation or None
        Observatory location.
    """
    
    def __init__(self, header_template_path=None, debug=False):
        """
        Initialize the header builder with a template .ini file.

        Parameters
        ----------
        header_template_path_ini : str
            Path to the configuration file containing the base header.
        """

        if debug:
            log.setLevel('DEBUG')
        
        self._coord = None
        self._location = None
        self._obstime = None

        self._head_dir = "headers"
        self._obs_dir = "observatories"
        
        self.header = fits.Header()
        self.load_default_header(header_template_path)
        # self.load_noctis_observatory("oarpaf")
        # self.set_coordinates(05.3, 70.0, Time.now().isot)

        
    @property
    def noctis_observatories(self):
        """
        List available [observatory].ini files in the module directory.

        Returns
        -------
        list of str
            Names of available observatory configuration files (without .ini extension).
        """
        data_dir = Path(__file__).parent / self._obs_dir
        return [f.stem for f in data_dir.glob("*.ini")]

        
    def load_default_header(self, path):
        """
        Load header from configuration file
        
        Parameters
        ----------
        path : str
            Configuration file in .ini format
        """

        if not path:
            path = Path(__file__).parent / self._head_dir / "header_base_v1.ini"
            log.info(path)
        
        config = configparser.ConfigParser(inline_comment_prefixes=('#',))
        try:
            config.read(path)
        except IndexError as e:
            log.error("File not found")
            
        for section in config.sections():
            for key, value in config.items(section):
                
                try:
                    val, comment = value.split("|")
                except ValueError:
                    # HISTORY and COMMENT have no value
                    val, comment = value, None

                val = self._parse(val)
                self.header[key] = val, comment

        self._update()        
        
        
    def load_noctis_observatory(self, name='oarpaf', flavor=None):
        data_dir = Path(__file__).parent / self._obs_dir
        ini_path = data_dir / f"{name}.ini"

        log.debug(ini_path)
        if not ini_path.exists():
            raise FileNotFoundError(f"Observatory config '{ini_path}' not found.")

        self.load_observatory(str(ini_path), flavor=flavor)
        
        
    def load_observatory(self, path=None, flavor=None):
        """
        Load observatory parameters such as location and detector specifics,
        and update relevant FITS keywords.
        
        Parameters
        ----------
        path : str
            Configuration file in .ini format.
        flavor : str, optional
            Section name of the configuration file. If not specified, takes the first one.
        """
        
        config = configparser.ConfigParser(inline_comment_prefixes=('#',))
        config.read(path)

        sections = config.sections()

        if not flavor:
            loc = config[sections[0]]
        else:
            loc = config[flavor]

        self.set_location(loc["OBS-LONG"], loc["OBS-LAT"], loc["OBS-ELEV"])
            
        for k in loc.keys():
            val = self._parse(loc[k])
            if k in self.header:
                self.header[k] = val

        self.header["DETSIZE"] = f'[1:{self.header["NAXIS1"]},1:{self.header["NAXIS2"]}]'
                
        self._update()

        
    def set_location(self, lon, lat, alt):
        """
        Location of the observatory, Separate so that it is
        possible to add a location without a config file

        Parameters
        ----------
        lon : float
            Longitude in degrees.
        lat : float
            Latitude in degrees.
        alt : float
            Elevation in meters.
        """
        
        self._location = EarthLocation(lon, lat, alt)
        
        self.header["OBS-LONG"] = lon
        self.header["OBS-LAT"] = lat
        self.header["OBS-ELEV"] = alt
        
        
    def set_obstime(self, obstime):
        """ Set observation time

        Parameters
        ----------
        obstime : str, astropy.time.Time
            Time of the observation.
        """
        
        time = Time(obstime)        
        self._obstime = time
        
        if self._coord != None:
            self._coord.obstime = time
        
        self.header['DATE'] = time.isot.split("T")[0]
        self.header['DATE-OBS'] = time.isot
        self.header['MJD-OBS'] = time.mjd

        self._update()
        

    def set_object(self, objname, update_coord=True, obstime=None):
        """
        Resolve object coordinates and set OBJECT keyword.
    
        Parameters
        ----------
        objname : str
            Name of the object to be resolved.
        update_coord : bool, optional
            If True, updates coordinate keywords.
        obstime : str or astropy.time.Time, optional
            Observation time to associate with coordinates.
        """
            
        if update_coord:
            try:
                coord = SkyCoord.from_name(objname)
                self._coord = coord

                if obstime:
                    time = Time(obstime)        
                    self.set_obstime(time)
                
                log.info("Found catalog name")
                coorstr = coord.to_string(style='hmsdms', sep=' ', precision=1, pad=True)
                log.info(f"Corresponds to {coorstr}")
                
                if obstime:
                    self.set_coordinates(coord.ra, coord.dec, obstime)
                else:
                    self.set_coordinates(coord.ra, coord.dec)
                    
            except NameResolveError as e:
                log.error("Cannot resolve name")
                return [None, None]
        
        self.header["OBJECT"] = objname

            
    def set_coordinates(self, ra, dec, obstime=None):
        """
        RA e DEC in decimal or sexagesimal format, and ISO observation time
        Set the object sky coordinates and compute related quantities.

        Parameters
        ----------
        ra : float
            Right Ascension in decimal degrees or hours.
        dec : float
            Declination in decimal degrees.
        obstime : str or astropy.time.Time, optional
            Observation time.
        """

        coord = SkyCoord(ra=ra, dec=dec, unit=(u.hourangle, u.deg))
        self._coord = coord

        if obstime:
            time = Time(obstime)
            self.set_obstime(time)
            
        self.header['RA'] = coord.ra.to_string(unit=u.hourangle, sep=':')
        self.header['DEC'] = coord.dec.to_string(unit=u.deg, sep=':')
        self.header['RA_DEG'] = coord.ra.deg
        self.header['DEC_DEG'] = coord.dec.deg

        self._update()

            
    def set_altaz_and_parallactic(self):
        """
        Compute and update Alt/Az, Airmass, LST, HA, Position and Parallactic Angle.
    
        Raises
        ------
        ValueError
            If observation time or location is not set.
        """
        
        if self._obstime == None or self._location == None:
            raise ValueError("Observation Time, Instrument parameters must be set.")
        
        self._coord.obstime = self._obstime
        self._coord.location = self._location
        altaz = self._coord.altaz

        # Altitudine and Azimuth
        self.header['ALT'] = round(altaz.alt.deg, 7)
        self.header['AZ'] = round(altaz.az.deg, 7)
        self.header['AIRMASS'] = round(altaz.secz.value, 2)

        # Local Sideral Time and Hour Angle
        lst = self._obstime.sidereal_time('mean', longitude=self._location.lon)
        ha = (lst - self._coord.ra).hour 
        lst_hours = lst.hour

        self.header['LST'] = round(lst_hours, 2)
        self.header['HA'] = round(ha, 2)

        # Position angle: with respect to Celestial North Pole
        north_celestial = SkyCoord(ra=0*u.deg, dec=90*u.deg, frame='icrs')
        posang = self._coord.position_angle(north_celestial).to(u.deg).value
        self.header['POSANGLE'] = round(posang, 2)

        # Parallactic angle:  between local meridian and celestial axis
        parangle = (posang - altaz.az.deg + 360) % 360
        if parangle > 180:
            parangle -= 360  # Wrap to [-180, 180]

        self.header['PARANGLE'] = round(parangle, 2)


    def set_wcs(self, angle=None):
        """
        Set World Coordinate System (WCS) parameters.
    
        Parameters
        ----------
        angle : float, optional
            Rotation angle in degrees. Defaults to DEROTANG.
    
        Raises
        ------
        ValueError
            If coordinates or location are not set.
        """
          
        if self._coord == None or self._location == None:
            raise ValueError("Observation Coordinates, Instrument parameters must be set.")

        crpix = [self.header['NAXIS1']/2, self.header['NAXIS2']/2]
        cdelt = [round(self.header["PIXSCALE"]*u.arcsec.to(u.deg), 7),
                 round(self.header["PIXSCALE"]*u.arcsec.to(u.deg), 7)]

        if not angle:
            angle = self.header["DEROTANG"]
            
        angle = np.deg2rad(angle)
            
        crval_ra = self._coord.ra.deg
        crval_dec = self._coord.dec.deg

        self.header['CRPIX1'] = crpix[0]
        self.header['CRPIX2'] = crpix[1]
        self.header['CRVAL1'] = crval_ra
        self.header['CRVAL2'] = crval_dec
        self.header['CDELT1'] = cdelt[0]
        self.header['CDELT2'] = cdelt[1]
        self.header["PC1_1"] = -np.cos(angle)
        self.header["PC2_1"] = +np.sin(angle)
        self.header["PC1_2"] = +np.sin(angle)
        self.header["PC2_2"] = +np.cos(angle)

        
    def set_ambient(self):
        """
        Set ambient keywords: Moon distance, phase and Sun altitude.
    
        Raises
        ------
        ValueError
            If coordinates or location are not set.
        """
            
        if self._coord == None or self._location == None:
            raise ValueError("Observation Time, Observing location must be set.")

        time = self._coord.obstime
        loc = self._location
        
        # Sun and Moon in alt-az coordinates
        with solar_system_ephemeris.set('builtin'):
            sun = get_body("sun", time)
            sun.location = loc
            moon = get_body("moon", time)
            moon.location = loc            
    
        # MOONDIST: angular distance between target and Moon
        moondist = moon.separation(self._coord, origin_mismatch="ignore").deg
        self.header['MOONDIST'] = round(moondist, 1)
    
        # MOONPHAS: Moon phase
        elongation = moon.separation(sun).deg
        moonphas = (1 + np.cos(np.radians(elongation))) / 2
        self.header['MOONPHAS'] = round(moonphas, 2)
    
        # SUNALT: Sun altitude above the horizon
        sunalt = sun.altaz.alt.deg
        self.header['SUNALT'] = round(sunalt, 1)

        
    def check_empty(self):
        """
        List the header keywords that are still empty
        """
        
        for k in self.header:
            if self.header[k] == None:
                print(k, self.header[k])

    
    def _update(self):
        """
        If coordinates, observation and location are provided, then
        additional keywords can be filled
        """

        if self._coord != None and self._obstime != None and self._location != None:
            self.set_altaz_and_parallactic()
            self.set_ambient()
            self.set_wcs()

            
    def _parse(self, val):
        """
        Parse values from configuration file.
    
        Parameters
        ----------
        val : str
            Value to be parsed.
    
        Returns
        -------
        int, float, bool, or str
            Parsed value.
        """
                   
        val = val.strip()
            
        if not val:
            val = None
        else:
            try:
                # float
                val = float(val)
                if  val.is_integer():
                    # int
                    val = int(val)
            except ValueError as e:
                # bool
                if val.lower() == "true":
                    val = True
                elif val.lower() == "false":
                    val = False
                else:
                    # string
                    pass
                
        return val

