
# Noche - Noctis Header. 
## FITS Header Builder for the observatories of the NOCTIS network

**Noche** is a Python module that helps construct FITS headers for
imaging (and maybe spectroscopic) observations using templates and
observatory-specific metadata.

It is developed for the NOCTIS - Network of Observing Networked
Observational Coordination of Telescopes for Teaching and Science
network.

https://noctis.inaf.it/en/index.html

## Features
- Load an empty base header from a default `.ini` file
- Set a NOCTIS observatory from default `.ini` files, or from a custom file;
- Resolve object coordinates via Simbad from name, or set coordinates;
- Automatically compute WCS, parallactic angle, airmass, and moon/sun-related metadata, etc.

## Installation

```bash
# Clone the repository
git clone https://www.ict.inaf.it/gitlab/noctis/noche.git

# Enter the directory
cd noche

# Optional: Install it locally if you want
pip install -e .
```

## Usage Example
```python
from noche import Noche
from astropy.time import Time

# Instantiate a Noche class. It loads by default an empty header
n = Noche() 

# Inspect with
print(n.header.tostring)

# Show which observatories belongs to the NOCTIS network
print(n.noctis_observatories)
# ['grt', 'opc', 'oarpaf', 'ogg', 'ossfoligno', 'abobservatory']

# Load the values specific of one of the NOCTIS observatories
n.load_noctis_observatory("oarpaf")

# You can inspect changes with
print(n.header.tostring)

# Load the values from a custom .ini file:
n.load_observatory("./noche/observatories/opc.ini")

# Set coordinates and observation time. 
n.set_coordinates("05 30 00.0", "70 01 01", Time.now().isot)

# Set an object name, 
# optionally resolving and update coordinates from Simbad, 
# optionally update observation time
n.set_object("Polaris", update_coord=True, obstime="2025-10-13T22:33:44.5")

# Update Wcs Keywords providing an angle:
n.set_wcs(45)

# List the header keywords that are still empty
n.check_empty()

```

---

## Class and Method Documentation

### `class Noche`
Builds and maintains an astropy FITS header for astronomical observations.

#### `__init__(self, header_template_path=None)`
Initializes a new FITS header.

**Parameters**
- `header_template_path` : str
  Path to the header configuration template. If `None`, use default.

#### `@property def noctis_observatories(self)`
ist available [observatory].ini files in the module directory.

#### `load_default_header(self, path)`
Loads the base header from a default or a custom `.ini` configuration file.

**Parameters**
- `path` : str
  Path to a config file in INI format.

#### `load_noctis_observatory(self, name='oarpaf', flavor=None)`
Loads observatory site data and updates header values accordingly.

**Parameters**
- `name` : str
  One of the supported observatories.
- `flavor` : str, optional
  Specific section in the file to load.

#### `load_observatory(self, path='oarpaf.ini', flavor=None)`
Loads observatory site data and updates header values accordingly.

**Parameters**
- `path` : str
  Path to the observatory INI file.
- `flavor` : str, optional
  Specific section in the file to load.

#### `set_location(self, lon, lat, alt)`
Sets the location of the observatory.

**Parameters**
- `lon` : float
  Longitude in degrees
- `lat` : float
  Latitude in degrees
- `alt` : float
  Altitude in meters

#### `set_obstime(self, obstime)`
Sets the observation time.

**Parameters**
- `obstime` : str or `astropy.time.Time`
  Time of the observation.

#### `set_object(self, objname, update_coord=True, obstime=None)`
Sets object name and optionally resolves coordinates via Simbad.

**Parameters**
- `objname` : str
  Name of the astronomical object.
- `update_coord` : bool, default=True
  Whether to resolve and update coordinates.
- `obstime` : str or `astropy.time.Time`, optional
  Time of the observation.

#### `set_coordinates(self, ra, dec, obstime=None)`
Sets the RA and DEC of the object.

**Parameters**
- `ra` : float
  Right Ascension (in hours or degrees).
- `dec` : float
  Declination (in degrees).
- `obstime` : str or `astropy.time.Time`, optional
  Time of the observation.

#### `set_altaz_and_parallactic(self)`
Calculates altitude, azimuth, airmass, hour angle, and parallactic angle.

**Raises**
- `ValueError` if observation time or location is not set.

#### `set_wcs(self, angle=None)`
Sets World Coordinate System (WCS) parameters in the header.

**Parameters**
- `angle` : float, optional
  Rotation angle in radians. Defaults to DEROTANG header value.

**Raises**
- `ValueError` if coordinates or location are missing.

#### `set_ambient(self)`
Adds information on Sun altitude, Moon phase, and Moon distance to target.

**Raises**
- `ValueError` if coordinates or location are missing.

#### `check_empty(self)`
Prints out FITS header keys that have `None` values.

#### `_update(self)`
Automatically fills in derived keywords if coordinates, time, and location are available.

#### `_parse(self, val)`
Parses a value from a configuration string.

**Parameters**
- `val` : str
  Value as string from config file.

**Returns**
- Parsed value as `int`, `float`, `bool`, or `str`.

---




## Getting started

To make it easy for you to get started with GitLab, here's a list of recommended next steps.

Already a pro? Just edit this README.md and make it your own. Want to make it easy? [Use the template at the bottom](#editing-this-readme)!

## Add your files

- [ ] [Create](https://docs.gitlab.com/ee/user/project/repository/web_editor.html#create-a-file) or [upload](https://docs.gitlab.com/ee/user/project/repository/web_editor.html#upload-a-file) files
- [ ] [Add files using the command line](https://docs.gitlab.com/topics/git/add_files/#add-files-to-a-git-repository) or push an existing Git repository with the following command:

```
cd existing_repo
git remote add origin https://www.ict.inaf.it/gitlab/noctis/noche.git
git branch -M main
git push -uf origin main
```

## Integrate with your tools

- [ ] [Set up project integrations](https://www.ict.inaf.it/gitlab/noctis/noche/-/settings/integrations)

## Collaborate with your team

- [ ] [Invite team members and collaborators](https://docs.gitlab.com/ee/user/project/members/)
- [ ] [Create a new merge request](https://docs.gitlab.com/ee/user/project/merge_requests/creating_merge_requests.html)
- [ ] [Automatically close issues from merge requests](https://docs.gitlab.com/ee/user/project/issues/managing_issues.html#closing-issues-automatically)
- [ ] [Enable merge request approvals](https://docs.gitlab.com/ee/user/project/merge_requests/approvals/)
- [ ] [Set auto-merge](https://docs.gitlab.com/user/project/merge_requests/auto_merge/)

## Test and Deploy

Use the built-in continuous integration in GitLab.

- [ ] [Get started with GitLab CI/CD](https://docs.gitlab.com/ee/ci/quick_start/)
- [ ] [Analyze your code for known vulnerabilities with Static Application Security Testing (SAST)](https://docs.gitlab.com/ee/user/application_security/sast/)
- [ ] [Deploy to Kubernetes, Amazon EC2, or Amazon ECS using Auto Deploy](https://docs.gitlab.com/ee/topics/autodevops/requirements.html)
- [ ] [Use pull-based deployments for improved Kubernetes management](https://docs.gitlab.com/ee/user/clusters/agent/)
- [ ] [Set up protected environments](https://docs.gitlab.com/ee/ci/environments/protected_environments.html)

***

## Badges
On some READMEs, you may see small images that convey metadata, such as whether or not all the tests are passing for the project. You can use Shields to add some to your README. Many services also have instructions for adding a badge.

