import unittest
from astropy.time import Time
from noche import Noche
from astropy.io.fits import Header

class TestNoche(unittest.TestCase):

    def setUp(self):
        self.noche = Noche("noche/headers/header_base.ini")
        self.noche.load_observatory("noche/observatories/oarpaf.ini")

    def test_init_creates_header(self):
        self.assertIsInstance(self.noche.header, Header)

    def test_set_location(self):
        self.noche.set_location(10.0, 45.0, 100.0)
        self.assertEqual(self.noche.header['OBS-LONG'], 10.0)
        self.assertEqual(self.noche.header['OBS-LAT'], 45.0)
        self.assertEqual(self.noche.header['OBS-ELEV'], 100.0)

    def test_set_obstime(self):
        now = Time.now()
        self.noche.set_obstime(now)
        self.assertEqual(self.noche.header['DATE-OBS'], now.isot)
        self.assertAlmostEqual(self.noche.header['MJD-OBS'], now.mjd, places=5)

    def test_set_coordinates(self):
        self.noche.set_coordinates(ra=10.0, dec=20.0)
        self.assertIn('RA', self.noche.header)
        self.assertIn('DEC', self.noche.header)
        self.assertAlmostEqual(self.noche.header['RA_DEG'], 150.0, places=4)  # 10h -> 150deg
        self.assertAlmostEqual(self.noche.header['DEC_DEG'], 20.0, places=4)

    def test_set_object_and_resolve(self):
        self.noche.set_object("Vega")
        self.assertEqual(self.noche.header['OBJECT'], "Vega")

    def test_check_empty_does_not_raise(self):
        # Should not raise even if empty fields exist
        try:
            self.noche.check_empty()
        except Exception as e:
            self.fail(f"check_empty() raised an exception: {e}")

    def test_parse_various_types(self):
        self.assertEqual(self.noche._parse("42"), 42)
        self.assertEqual(self.noche._parse("3.14"), 3.14)
        self.assertEqual(self.noche._parse("True"), True)
        self.assertEqual(self.noche._parse("False"), False)
        self.assertEqual(self.noche._parse("some string"), "some string")
        self.assertIsNone(self.noche._parse("   "))

if __name__ == '__main__':
    unittest.main()
