#include "iteration_functions.h"

/*==================================================================*/

double Ak_integral(double x, void* params)
{
  gsl_set_error_handler_off();

  int seed_location = ((AngularFunctionParams*)params)->seed_location;
  int k = ((AngularFunctionParams*)params)->k;
  int status;
  double tau = ((AngularFunctionParams*)params)->tau;
  double tau_s = ((AngularFunctionParams*)params)->tau_s;

  double tau_round_high;

  double Ikl_a, Ikl_b, value;

  if (k == 1)
  {
    if (seed_location == SEED_CENTER)
    {
      Ikl_a = I0_center(tau0, tau, x);
      Ikl_b = I0_center(tau0, 2 * tau0 - tau, x);
      // printf("Center: Tau %lf 2tau0-tau %5.4f I(tau)=%5.4e I(2tau0-tau)=%5.4e  (u=%lf) \n", tau,
      //                        2 * tau0 - tau, Ikl_a, Ikl_b, x);
    }
    else if (seed_location == SEED_UNIFORM)
    {
      Ikl_a = I0_uniform(tau0, tau, x);
      Ikl_b = I0_uniform(tau0, 2 * tau0 - tau, x);
    }
    else if (seed_location == SEED_DELTA)
    {
      Ikl_a = I0_delta(tau, tau_s, x);
      Ikl_b = I0_delta(2 * tau0 - tau, 2 * tau0 - tau_s, x);

      printf("Delta: Tau %lf 2tau0-tau %5.4f I(tau)=%5.4e I(2tau0-tau)=%5.4e  (u=%lf) \n", tau,
             2 * tau0 - tau, Ikl_a, Ikl_b, x);
    }
    else if (seed_location == SEED_BASE)
    {
      Ikl_a = I0_base(tau, tau_s, x, U_POSITIVE);
      Ikl_b = I0_base(2 * tau0 - tau, tau_s, x, U_NEGATIVE);

      printf("Base: Tau %lf 2tau0-tau %5.4f I(tau)=%5.4e I(2tau0-tau)=%5.4e  (u=%lf) \n", tau,
             2 * tau0 - tau, Ikl_a, Ikl_b, x);
    }
    else if (seed_location == SEED_EXP)
    {
      Ikl_a = I0_exp(tau, tau0, x);
      Ikl_b = I0_exp(2 * tau0 - tau, tau0, x);
    }

    else
    {
      Ikl_a = 1;
      Ikl_b = 1;
    }
  }
  else
  {
    if (2 * tau0 - tau == 2 * tau0)
    {
      tau_round_high = 2 * tau0 - tau - 1e-7;
    }
    else
    {
      tau_round_high = 2 * tau0 - tau;
    }

    status = gsl_spline2d_eval_e(Spline_I2d_l, tau, x, xacc_2d, yacc_2d, &Ikl_a);

    if (status)
    {
      printf("Source file %s\n", __FILE__);
      printf("u=%lf tau=%lf\n", x, tau);
      printf("Line %d Error: %s\n", __LINE__, gsl_strerror(status));
      exit(1);
    }

    status = gsl_spline2d_eval_e(Spline_I2d_l, tau_round_high, x, xacc_2d, yacc_2d, &Ikl_b);

    if (status)
    {
      printf("error: %s\n", gsl_strerror(status));
    }
  }

  value = 3. / 4 * (1 - x * x) * (Ikl_a + Ikl_b);

  return value;
}

/*==================================================================*/

double Bk_integral(double x, void* params)
{
  int status;
  int seed_location = ((AngularFunctionParams*)params)->seed_location;
  int k = ((AngularFunctionParams*)params)->k;
  double tau = ((AngularFunctionParams*)params)->tau;
  double tau_s = ((AngularFunctionParams*)params)->tau_s;
  double tau_round_high;

  double Ikl_a, Ikl_b, value;
  if (k == 1)
  {
    if (seed_location == SEED_CENTER)
    {
      Ikl_a = I0_center(tau0, tau, x);
      Ikl_b = I0_center(tau0, 2 * tau0 - tau, x);
    }
    else if (seed_location == SEED_UNIFORM)
    {
      Ikl_a = I0_uniform(tau0, tau, x);
      Ikl_b = I0_uniform(tau0, 2 * tau0 - tau, x);
    }
    else if (seed_location == SEED_DELTA)
    {
      Ikl_a = I0_delta(tau, tau_s, x);
      Ikl_b = I0_delta(2 * tau0 - tau, 2 * tau0 - tau_s, x);
    }
    else if (seed_location == SEED_BASE)
    {
      Ikl_a = I0_base(tau, tau_s, x, U_POSITIVE);
      Ikl_b = I0_base(2 * tau0 - tau, tau_s, x, U_NEGATIVE);
    }
    else if (seed_location == SEED_EXP)
    {
      Ikl_a = I0_exp(tau, tau0, x);
      Ikl_b = I0_exp(2 * tau0 - tau, tau0, x);
    }

    else
    {
      Ikl_a = 1;
      Ikl_b = 1;
    }
  }
  else
  {
    if (2 * tau0 - tau == 2 * tau0)
    {
      tau_round_high = 2 * tau0 - tau - 1e-7;
    }
    else
    {
      tau_round_high = 2 * tau0 - tau;
    }

    status = gsl_spline2d_eval_e(Spline_I2d_l, tau, x, xacc_2d, yacc_2d, &Ikl_a);

    if (status)
    {
      printf("error: %s\n", gsl_strerror(status));
    }

    status = gsl_spline2d_eval_e(Spline_I2d_l, tau_round_high, x, xacc_2d, yacc_2d, &Ikl_b);

    if (status)
    {
      printf("error: %s\n", gsl_strerror(status));
    }
  }

  value = 3. / 8 * pow(x, 2) * (Ikl_a + Ikl_b);

  return value;
}

double Ck_integral(double x, void* params)
{
  int status;
  int seed_location = ((AngularFunctionParams*)params)->seed_location;
  int k = ((AngularFunctionParams*)params)->k;
  double tau = ((AngularFunctionParams*)params)->tau;
  double tau_s = ((AngularFunctionParams*)params)->tau_s;
  double tau_round_high;

  double Ikr_a, Ikr_b, value;

  if (k == 1)
  {
    if (seed_location == SEED_CENTER)
    {
      Ikr_a = I0_center(tau0, tau, x);
      Ikr_b = I0_center(tau0, 2 * tau0 - tau, x);
    }
    else if (seed_location == SEED_UNIFORM)
    {
      Ikr_a = I0_uniform(tau0, tau, x);
      Ikr_b = I0_uniform(tau0, 2 * tau0 - tau, x);
    }
    else if (seed_location == SEED_DELTA)
    {
      Ikr_a = I0_delta(tau, tau_s, x);
      Ikr_b = I0_delta(2 * tau0 - tau, 2 * tau0 - tau_s, x);
    }
    else if (seed_location == SEED_BASE)
    {
      Ikr_a = I0_base(tau, tau_s, x, U_POSITIVE);
      Ikr_b = I0_base(2 * tau0 - tau, 2 * tau0 - tau_s, x, U_NEGATIVE);
    }
    else if (seed_location == SEED_EXP)
    {
      Ikr_a = I0_exp(tau, tau_s, x);
      Ikr_b = I0_exp(2 * tau0 - tau, tau0, x);
    }

    else
    {
      Ikr_a = 1;
      Ikr_b = 1;
    }
  }
  else
  {
    if (2 * tau0 - tau == 2 * tau0)
    {
      tau_round_high = 2 * tau0 - tau - 1e-8;
    }
    else
    {
      tau_round_high = 2 * tau0 - tau;
    }

    status = gsl_spline2d_eval_e(Spline_I2d_r, tau, x, xacc_2d, yacc_2d, &Ikr_a);

    if (status)
    {
      printf("error: %s\n", gsl_strerror(status));
    }

    status = gsl_spline2d_eval_e(Spline_I2d_r, tau_round_high, x, xacc_2d, yacc_2d, &Ikr_b);

    if (status)
    {
      printf("error: %s\n", gsl_strerror(status));
    }
  }

  value = 3. / 8 * (Ikr_a + Ikr_b);

  return value;
}

/*==================================================================*/
double Il_integral(double x, void* params)
{
  double Ak_interp, Bk_interp, Ck_interp;
  double value, tau, mu;

  tau = ((double*)params)[0];
  mu = ((double*)params)[1];

  Ak_interp = gsl_spline_eval(Spline_tau_Ak, x, xacc_1d);
  Bk_interp = gsl_spline_eval(Spline_tau_Bk, x, xacc_1d);
  Ck_interp = gsl_spline_eval(Spline_tau_Ck, x, xacc_1d);

  value = exp(-(x - tau) / mu) * ((1 - mu * mu) * Ak_interp + mu * mu * (Bk_interp + Ck_interp)) * 1 / mu;

  return value;
}

/*==================================================================*/
double Ir_integral(double x, void* params)
{
  double Bk_interp, Ck_interp;
  double value, tau, mu;

  tau = ((double*)params)[0];
  mu = ((double*)params)[1];

  Bk_interp = gsl_spline_eval(Spline_tau_Bk, x, xacc_1d);
  Ck_interp = gsl_spline_eval(Spline_tau_Ck, x, xacc_1d);

  value = exp(-(x - tau) / mu) * (Bk_interp + Ck_interp) * 1 / mu;

  return value;
}
