/*
 * This file is part of vospace-datamodel
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace.datamodel;

import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import java.io.IOException;
import java.util.Map;
import net.ivoa.xml.uws.v1.JobSummary.JobInfo;
import net.ivoa.xml.vospace.v2.Transfer;

public class JobInfoDeserializer extends StdDeserializer<JobInfo> {

    private static final ObjectMapper MAPPER = new ObjectMapper();

    public JobInfoDeserializer() {
        super(JobInfo.class);
    }

    @Override
    public JobInfo deserialize(JsonParser jp, DeserializationContext dc) throws IOException, JsonProcessingException {
        Object content = jp.getCodec().readValue(jp, Object.class);

        if (content == null) {
            return null;
        }

        if (!(content instanceof Map)) {
            throw new UnsupportedOperationException("JobInfo contains an instance of " + content.getClass().getCanonicalName());
        }

        Map<String, Object> map = (Map<String, Object>) content;

        if (map.isEmpty()) {
            return null;
        }
        if (map.keySet().size() > 1) {
            throw new UnsupportedOperationException("Multiple keys found in JobInfo content");
        }

        String name = map.keySet().toArray(String[]::new)[0];

        JobInfo jobInfo = new JobInfo();

        switch (name) {
            case "transfer":
                String transferJson = MAPPER.writeValueAsString(map.get(name));
                Transfer transfer = MAPPER.readValue(transferJson, Transfer.class);
                jobInfo.getAny().add(transfer);
                break;
            default:
                throw new UnsupportedOperationException("JobInfo map key is " + name);
        }

        return jobInfo;
    }
}
