/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.service;

import it.inaf.ia2.transfer.persistence.FileDAO;
import it.inaf.ia2.transfer.persistence.model.FileInfo;
import it.inaf.oats.vospace.exception.QuotaExceededException;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.nio.ByteBuffer;
import java.nio.channels.FileChannel;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.LocalDateTime;
import java.util.Objects;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.xml.bind.DatatypeConverter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class PutFileService {

    private static final Logger LOG = LoggerFactory.getLogger(PutFileService.class);

    @Autowired
    private FileDAO fileDAO;

    public synchronized void makeFoldersPath(File file) {
        /**
         * This method must be synchronized, to avoid concurrency issues when
         * multiple files are uploaded to a new folder in parallel.
         */
        if (!file.getParentFile().exists()) {
            if (!file.getParentFile().mkdirs()) {
                throw new IllegalStateException("Unable to create parent folder: " + file.getParentFile().getAbsolutePath());
            }
        }
    }

    public void copyLocalFile(FileInfo sourceFileInfo, FileInfo destinationFileInfo, Long remainingQuota) {

        File destinationFile = this.prepareDestination(destinationFileInfo);
        File sourceFile = new File(sourceFileInfo.getFilePath());

        try {
            Files.copy(sourceFile.toPath(), destinationFile.toPath());
            this.finalizeFile(sourceFileInfo, destinationFileInfo, destinationFile, remainingQuota);
        } catch (IOException e) {
            destinationFile.delete();
            throw new UncheckedIOException(e);
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
    }

    public void storeFileFromInputStream(FileInfo destinationFileInfo,
            InputStream is, Long remainingQuota) {
        this.storeFileFromInputStream(null, destinationFileInfo, is, remainingQuota);
    }

    public void storeFileFromInputStream(FileInfo sourceFileInfo, FileInfo destinationFileInfo,
            InputStream is, Long remainingQuota) {

        File destinationFile = this.prepareDestination(destinationFileInfo);

        try {
            Files.copy(is, destinationFile.toPath());
            this.finalizeFile(sourceFileInfo, destinationFileInfo, destinationFile, remainingQuota);
        } catch (IOException e) {
            destinationFile.delete();
            throw new UncheckedIOException(e);
        } catch (NoSuchAlgorithmException e) {
            destinationFile.delete();
            throw new RuntimeException(e);
        }

    }

    private File prepareDestination(FileInfo destinationFileInfo) {
        
        // TODO: this reproduces past behaviour, we need to confirm 
        // basePath null case and if we want to lock the node after 
        // the first upload (fsPath not null)
        if(destinationFileInfo.getActualBasePath() != null) {
            if(destinationFileInfo.getFsPath() != null) {
                LOG.warn("Node {} fsPath is not null: {}. Overwriting.", 
                        destinationFileInfo.getVirtualPath(), 
                        destinationFileInfo.getFsPath());
            }
            
            destinationFileInfo.setFsPath(this.generateFsPath().toString());                       
                        
        }
                   
        File file = new File(destinationFileInfo.getFilePath());    
        makeFoldersPath(file);
                       
        // This is EXTREMELY unlikely to happen, but we want to be safe here
        if(file.exists()) {
            throw new IllegalStateException("Unexpected collision for generated file path: " 
                    + file.getPath());           
        }

        return file;
    }

    private Long checkQuota(File destinationFile, Long remainingQuota) throws IOException {
        Long fileSize = Files.size(destinationFile.toPath());

        // Quota limit is checked again to handle cases where MultipartFile is not used
        if (remainingQuota != null && fileSize > remainingQuota) {
            destinationFile.delete();
            throw new QuotaExceededException("Path: " + destinationFile.toPath().toString());
        }

        return fileSize;
    }

    private void finalizeFile(FileInfo sourceFileInfo, FileInfo destinationFileInfo,
            File destinationFile, Long remainingQuota) throws IOException, NoSuchAlgorithmException {

        Long fileSize = this.checkQuota(destinationFile, remainingQuota);

        if (destinationFileInfo.getContentType() == null) {
            destinationFileInfo.setContentType(Files.probeContentType(destinationFile.toPath()));
        }

        String md5Checksum = makeMD5Checksum(destinationFile);

        // TODO: discuss if mismatches lead to taking actions
        if (sourceFileInfo != null) {
            if (!Objects.equals(sourceFileInfo.getContentLength(), fileSize)) {
                LOG.warn("Destination file {} size mismatch with source", destinationFile.toPath().toString());
            }

            if (sourceFileInfo.getContentType() != null
                    && !sourceFileInfo.getContentType().equals(destinationFileInfo.getContentType())) {
                LOG.warn("Destination file {} content type mismatch with source {} {}", destinationFile.toPath().toString(),
                        destinationFileInfo.getContentType(), sourceFileInfo.getContentType());
            }

            if (sourceFileInfo.getContentMd5() != null
                    && !sourceFileInfo.getContentMd5().equals(md5Checksum)) {
                LOG.warn("Destination file {} md5 mismatch with source {} {}", destinationFile.toPath().toString(),
                        destinationFileInfo.getContentMd5(), sourceFileInfo.getContentMd5());
            }

        }

        fileDAO.updateFileAttributes(destinationFileInfo.getNodeId(),
                destinationFileInfo.getFsPath(),
                destinationFileInfo.getContentType(),
                destinationFileInfo.getContentEncoding(),
                fileSize,
                md5Checksum);
    }

    /**
     * Handles duplicate file uploads generating a new non existent path. This
     * is necessary in some edge cases, like when a file has been renamed in
     * VOSpace only but the original file on disk still has the old name or if a
     * file has been marked for deletion and a file with the same name is
     * uploaded before the cleanup.
     */
    private File getEmptyFile(File file, int index) {
        if (file.exists()) {

            String fileName = file.getName();

            String nameWithoutExtension;
            String extension = null;
            if (fileName.contains(".")) {
                nameWithoutExtension = fileName.substring(0, fileName.lastIndexOf("."));
                extension = fileName.substring(fileName.lastIndexOf(".") + 1, fileName.length());
            } else {
                nameWithoutExtension = fileName;
            }

            Pattern pattern = Pattern.compile("(.*?)-(\\d+)");
            Matcher matcher = pattern.matcher(nameWithoutExtension);
            if (matcher.matches()) {
                nameWithoutExtension = matcher.group(1);
                int fileIndex = Integer.parseInt(matcher.group(2));
                index = fileIndex + 1;
            }

            String newName = nameWithoutExtension + "-" + index;
            if (extension != null) {
                newName += "." + extension;
            }

            File newFile = file.toPath().getParent().resolve(newName).toFile();
            return getEmptyFile(newFile, index + 1);
        }
        return file;
    }

    private String makeMD5Checksum(File file) throws NoSuchAlgorithmException, IOException {
        MessageDigest md = MessageDigest.getInstance("MD5");

        // We can't update the MessageDigest object in a single step using
        // Files.readAllBytes because we want to handle also big files and that
        // method loads all the content in memory (OutOfMemoryError has been
        // noticed for files bigger than 1GB). So multiple updates using
        // FileChannel and ByteBuffer are used
        try (FileChannel channel = FileChannel.open(file.toPath(), StandardOpenOption.READ)) {
            // creating buffer (by default it is in write mode). FileChannel will write into buffer
            ByteBuffer buffer = ByteBuffer.allocate(1024);
            int bytesRead;
            while ((bytesRead = channel.read(buffer)) != -1) {
                if (bytesRead > 0) {
                    // switch from write to read mode
                    buffer.flip();
                    md.update(buffer);
                }
                // using rewind the buffer can be filled again
                buffer.rewind();
            }
        }

        byte[] digest = md.digest();
        String checksum = DatatypeConverter.printHexBinary(digest);
        return checksum;
    }

    private Path generateFsPath() {
        // Generate date and time part        
        LocalDateTime now = LocalDateTime.now();

        Path fsPath = Path.of(Integer.toString(now.getYear()))
                .resolve(Integer.toString(now.getMonthValue()))
                .resolve(Integer.toString(now.getDayOfMonth()));
        
        // Generate uuid for filename
        UUID uuid = UUID.randomUUID();
              
        fsPath = fsPath.resolve(uuid.toString());
        
        return fsPath;
    }

}
