package it.inaf.ia2.transfer.controller;

import it.inaf.ia2.transfer.persistence.FileDAO;
import it.inaf.ia2.transfer.persistence.ListOfFilesDAO;
import java.io.BufferedReader;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import javax.servlet.http.HttpServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import static org.springframework.http.HttpStatus.NOT_FOUND;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.PutMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.multipart.MultipartFile;

@RestController
public class PutFileController {

    @Autowired
    private FileDAO fileDAO;

    @Autowired
    private ListOfFilesDAO listOfFilesDAO;

    @Autowired
    private HttpServletRequest request;

    @PutMapping("/**")
    public ResponseEntity<?> putFile(@RequestParam("file") MultipartFile file) throws IOException {

        String path = request.getServletPath();
        Optional<FileInfo> optFileInfo = fileDAO.getFileInfo(path);

        if (optFileInfo.isPresent()) {
            try (InputStream in = file.getInputStream()) {
                storeFile(optFileInfo.get(), in);
            }
            return ResponseEntity.ok().build();
        } else {
            return new ResponseEntity<>("File " + path + " not found", NOT_FOUND);
        }
    }

    private void storeFile(FileInfo fileInfo, InputStream is) throws IOException {

        if (fileInfo.getAcceptViews() != null && fileInfo.getAcceptViews().contains("urn:list-of-files")) {
            storeListOfFiles(fileInfo, is);
        } else {
            storeGenericFile(fileInfo, is);
        }
    }

    private void storeListOfFiles(FileInfo fileInfo, InputStream is) throws IOException {
        List<String> filePaths = parseListOfFiles(is);
        listOfFilesDAO.createList(fileInfo.getVirtualPath(), filePaths);
    }

    private List<String> parseListOfFiles(InputStream is) throws IOException {
        List<String> filePaths = new ArrayList<>();
        try (BufferedReader br = new BufferedReader(new InputStreamReader(is))) {
            String line;
            while ((line = br.readLine()) != null && !line.isBlank()) {
                filePaths.add(line.trim());
            }
        }
        return filePaths;
    }

    private void storeGenericFile(FileInfo fileInfo, InputStream is) throws IOException {

        File file = new File(fileInfo.getOsPath());

        /**
         * This block must be synchronized, to avoid concurrency issues when
         * multiple files are uploaded to a new folder in parallel.
         */
        synchronized (this) {
            if (!file.getParentFile().exists()) {
                if (!file.getParentFile().mkdirs()) {
                    throw new IllegalStateException("Unable to create parent folder: " + file.getParentFile().getAbsolutePath());
                }
            }
        }

        String originalFileName = file.getName();
        file = getEmptyFile(file, 1);
        if (!originalFileName.equals(file.getName())) {
            fileDAO.setOsName(fileInfo.getNodeId(), file.getName());
        }

        try {
            fileDAO.setBusy(fileInfo.getNodeId(), true);
            Files.copy(is, file.toPath());
        } catch (IOException ex) {
            throw ex;
        } finally {
            fileDAO.setBusy(fileInfo.getNodeId(), false);
        }
    }

    /**
     * Handles duplicate file uploads generating a new non existent path. This
     * is necessary in some edge cases, like when a file has been renamed in
     * VOSpace only but the original file on disk still has the old name or if a
     * file has been marked for deletion and a file with the same name is
     * uploaded before the cleanup.
     */
    private File getEmptyFile(File file, int index) {
        if (file.exists()) {

            String fileName = file.getName();

            String nameWithoutExtension;
            String extension = null;
            if (fileName.contains(".")) {
                nameWithoutExtension = fileName.substring(0, fileName.lastIndexOf("."));
                extension = fileName.substring(fileName.lastIndexOf(".") + 1, fileName.length());
            } else {
                nameWithoutExtension = fileName;
            }

            String newName = nameWithoutExtension + "-" + index;
            if (extension != null) {
                newName += "." + extension;
            }

            File newFile = file.toPath().getParent().resolve(newName).toFile();
            return getEmptyFile(newFile, index + 1);
        }
        return file;
    }
}
