package it.inaf.ia2.transfer.persistence;

import it.inaf.ia2.transfer.controller.FileInfo;
import java.nio.file.Path;
import java.sql.Array;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import javax.sql.DataSource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Repository;

@Repository
public class FileDAO {

    @Value("${upload_location_id}")
    private int uploadLocationId;

    private final JdbcTemplate jdbcTemplate;

    @Autowired
    public FileDAO(DataSource fileCatalogDatasource) {
        this.jdbcTemplate = new JdbcTemplate(fileCatalogDatasource);
    }

    public Optional<FileInfo> getFileInfo(String virtualPath) {

        String sql = "SELECT n.node_id, is_public, group_read, group_write, owner_id, async_trans,\n"
                + "accept_views, provide_views, l.location_type,\n"
                + "(SELECT user_name FROM users WHERE rap_id = owner_id) AS username,\n"
                + "base_path, os_path\n"
                + "FROM node_path p\n"
                + "JOIN node n ON p.node_id = n.node_id\n"
                + "JOIN location l ON (n.location_id IS NOT NULL AND n.location_id = l.location_id) OR (n.location_id IS NULL AND l.location_id = ?)\n"
                + "LEFT JOIN storage s ON s.storage_id = l.storage_dest_id\n"
                + "WHERE p.vos_path = ?";

        FileInfo fileInfo = jdbcTemplate.query(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setInt(1, uploadLocationId);
            ps.setString(2, virtualPath);
            return ps;
        }, rs -> {
            if (rs.next()) {
                FileInfo fi = new FileInfo();
                fi.setNodeId(rs.getInt("node_id"));
                fi.setIsPublic(rs.getBoolean("is_public"));
                fi.setGroupRead(toList(rs.getArray("group_read")));
                fi.setGroupWrite(toList(rs.getArray("group_write")));
                fi.setOwnerId(rs.getString("owner_id"));
                fi.setAsyncTrans(rs.getBoolean("async_trans"));
                fi.setAcceptViews(toList(rs.getArray("accept_views")));
                fi.setProvideViews(toList(rs.getArray("provide_views")));
                fi.setVirtualPath(virtualPath);

                fillOsPath(fi, rs);

                return fi;
            }
            return null;
        });

        return Optional.ofNullable(fileInfo);
    }

    private void fillOsPath(FileInfo fi, ResultSet rs) throws SQLException {

        String basePath = rs.getString("base_path");
        String osPath = rs.getString("os_path");
        if (osPath.startsWith("/")) {
            osPath = osPath.substring(1);
        }

        Path completeOsPath = Path.of(basePath);

        boolean asyncLocation = "async".equals(rs.getString("location_type"));

        if (asyncLocation) {
            String username = rs.getString("username");
            completeOsPath = completeOsPath.resolve(username).resolve("retrieve");
        } else {
            completeOsPath = completeOsPath.resolve(fi.getOwnerId());
        }

        completeOsPath = completeOsPath.resolve(osPath);
        fi.setOsPath(completeOsPath.toString());
    }

    private List<String> toList(Array array) throws SQLException {
        if (array == null) {
            return new ArrayList<>();
        }
        return Arrays.asList((String[]) array.getArray());
    }

    public void setBusy(int nodeId, boolean busy) {

        String sql = "UPDATE node SET busy_state = ? WHERE node_id = ?";

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setBoolean(1, busy);
            ps.setInt(2, nodeId);
            return ps;
        });
    }

    public void setOsName(int nodeId, String newName) {

        String sql = "UPDATE node SET os_name = ? WHERE node_id = ?";

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            ps.setString(1, newName);
            ps.setInt(2, nodeId);
            return ps;
        });
    }
}
