package it.inaf.ia2.transfer.controller;

import it.inaf.ia2.transfer.persistence.FileDAO;
import it.inaf.ia2.transfer.persistence.ListOfFilesDAO;
import java.io.File;
import java.nio.charset.StandardCharsets;
import java.nio.file.Path;
import java.util.Collections;
import java.util.Optional;
import java.util.UUID;
import org.assertj.core.util.Files;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.mock.web.MockHttpServletRequest;
import org.springframework.mock.web.MockMultipartFile;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.request.MockMultipartHttpServletRequestBuilder;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import org.springframework.test.web.servlet.request.RequestPostProcessor;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@AutoConfigureMockMvc
@TestPropertySource(properties = {"path_prefix=/tmp"})
public class PutFileControllerTest {

    @MockBean
    private FileDAO fileDao;

    @MockBean
    private ListOfFilesDAO listOfFilesDAO;

    @Autowired
    private MockMvc mockMvc;

    @Test
    public void putGenericFile() throws Exception {

        String randomFileName = UUID.randomUUID().toString();
        createBaseFileInfo(randomFileName);

        MockMultipartFile fakeFile = new MockMultipartFile("file", "test.txt", "text/plain", "content".getBytes());

        mockMvc.perform(putMultipart("/path/to/test.txt")
                .file(fakeFile))
                .andDo(print())
                .andExpect(status().isOk());

        File file = Path.of("/tmp", randomFileName).toFile();

        assertTrue(file.exists());
        assertEquals("content", Files.contentOf(file, StandardCharsets.UTF_8));
        assertTrue(file.delete());
    }

    @Test
    public void putListOfFiles() throws Exception {

        String randomFileName = UUID.randomUUID().toString();
        FileInfo fileInfo = createBaseFileInfo(randomFileName);
        fileInfo.setAcceptViews(Collections.singletonList("urn:list-of-files"));

        String content = "/path/to/file1\n/path/to/file2";

        MockMultipartFile fakeFile = new MockMultipartFile("file", "test.txt", "text/plain", content.getBytes());

        mockMvc.perform(putMultipart("/path/to/test.txt")
                .file(fakeFile))
                .andDo(print())
                .andExpect(status().isOk());

        verify(listOfFilesDAO, times(1)).createList(any(), argThat(list -> list.size() == 2));
    }

    private FileInfo createBaseFileInfo(String fileName) {
        FileInfo fileInfo = new FileInfo();
        fileInfo.setOsRelPath(fileName);
        fileInfo.setIsPublic(false);

        when(fileDao.getFileInfo(any())).thenReturn(Optional.of(fileInfo));

        return fileInfo;
    }

    private MockMultipartHttpServletRequestBuilder putMultipart(String uri) {
        MockMultipartHttpServletRequestBuilder builder = MockMvcRequestBuilders.multipart(uri);
        builder.with(new RequestPostProcessor() {
            @Override
            public MockHttpServletRequest postProcessRequest(MockHttpServletRequest request) {
                request.setMethod("PUT");
                return request;
            }
        });
        return builder;
    }
}
