/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.controller;

import it.inaf.ia2.transfer.persistence.model.FileInfo;
import it.inaf.ia2.transfer.auth.TokenPrincipal;
import it.inaf.ia2.transfer.persistence.FileDAO;
import it.inaf.ia2.transfer.service.AuthorizationService;
import java.io.File;
import java.util.Optional;
import javax.servlet.http.HttpServletResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import static org.springframework.http.HttpStatus.NOT_FOUND;
import static org.springframework.http.HttpStatus.UNAUTHORIZED;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class GetFileController extends FileController {

    private static final Logger LOG = LoggerFactory.getLogger(GetFileController.class);

    @Autowired
    private FileDAO fileDAO;

    @Autowired
    private AuthorizationService authorizationService;

    @Autowired
    private HttpServletResponse response;

    @GetMapping("/**")
    public ResponseEntity<?> getFile() {

        String path = getPath();

        LOG.debug("getFile called for path {}", path);

        Optional<FileInfo> optFileInfo = fileDAO.getFileInfo(path);

        if (optFileInfo.isPresent()) {

            FileInfo fileInfo = optFileInfo.get();

            if (!authorizationService.isDownloadable(fileInfo, (TokenPrincipal) request.getUserPrincipal())) {
                return new ResponseEntity<>("Unauthorized", UNAUTHORIZED);
            }

            return getFileResponse(fileInfo);
        } else {
            return new ResponseEntity<>("File " + path + " not found", NOT_FOUND);
        }
    }

    private ResponseEntity<?> getFileResponse(FileInfo fileInfo) {

        File file = new File(fileInfo.getOsPath());

        return FileResponseUtil.getFileResponse(response, file, fileInfo.getVirtualName());
    }
}
