/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.service;

import it.inaf.ia2.transfer.auth.TokenPrincipal;
import it.inaf.ia2.transfer.persistence.FileDAO;
import it.inaf.ia2.transfer.persistence.JobDAO;
import it.inaf.ia2.transfer.persistence.LocationDAO;
import it.inaf.ia2.transfer.persistence.model.FileInfo;
import it.inaf.ia2.transfer.persistence.model.JobException;
import it.inaf.ia2.transfer.persistence.model.JobException.Type;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.security.Principal;
import java.util.List;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;
import javax.annotation.PostConstruct;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import org.kamranzafar.jtar.TarEntry;
import org.kamranzafar.jtar.TarOutputStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.stereotype.Service;
import org.springframework.util.FileSystemUtils;
import org.springframework.web.client.RestTemplate;

@Service
public class ArchiveService {

    private static final Logger LOG = LoggerFactory.getLogger(ArchiveService.class);

    @Autowired
    private FileDAO fileDAO;

    @Autowired
    private LocationDAO locationDAO;

    @Autowired
    private JobDAO jobDAO;

    @Autowired
    private AuthorizationService authorizationService;

    @Autowired
    private RestTemplate restTemplate;

    @Value("${upload_location_id}")
    private int uploadLocationId;

    @Value("${generated.dir}")
    private String generatedDirString;
    private File generatedDir;

    @PostConstruct
    public void init() {
        this.generatedDir = new File(generatedDirString);
        if (!this.generatedDir.exists()) {
            if (!this.generatedDir.mkdirs()) {
                throw new IllegalStateException("Unable to create directory " + this.generatedDir.getAbsolutePath());
            }
        }
    }

    public <O extends OutputStream, E> void createArchive(ArchiveJob job) {

        jobDAO.updateJobPhase(ExecutionPhase.EXECUTING, job.getJobId());

        LOG.trace("Started archive job " + job.getJobId());

        try {
            // TODO: check total size limit
            File archiveFile = getArchiveFile(job);

            String commonParent = getCommonParent(job.getVosPaths());
            // support directory used to generate folder inside tar files (path is redefined each time by TarEntry class)
            File supportDir = Files.createTempDirectory("dir").toFile();

            // it will be initialized only when necessary
            Map<Integer, String> portalLocationUrls = null;

            try ( ArchiveHandler<O, E> handler = getArchiveHandler(archiveFile, job.getType())) {

                for (FileInfo fileInfo : fileDAO.getArchiveFileInfos(job.getVosPaths())) {

                    String relPath = fileInfo.getVirtualPath().substring(commonParent.length());

                    if (fileInfo.isDirectory()) {
                        handler.putNextEntry(supportDir, relPath);
                        continue;
                    }

                    if (fileInfo.getLocationId() != null && fileInfo.getLocationId() != uploadLocationId) {
                        // remote file
                        if (portalLocationUrls == null) {
                            portalLocationUrls = locationDAO.getPortalLocationUrls();
                        }
                        String url = portalLocationUrls.get(fileInfo.getLocationId());
                        downloadFileIntoArchive(fileInfo, relPath, job.getPrincipal(), handler, url);
                    } else {
                        // local file or virtual directory
                        writeFileIntoArchive(fileInfo, relPath, job.getPrincipal(), handler);
                    }
                }
            } finally {
                FileSystemUtils.deleteRecursively(supportDir);
            }

            jobDAO.updateJobPhase(ExecutionPhase.COMPLETED, job.getJobId());

        } catch (Throwable t) {
            JobException jobException;
            if (t instanceof JobException) {
                jobException = (JobException) t;
            } else {
                LOG.error("Unexpected error happened creating archive", t);
                jobException = new JobException(Type.FATAL).setErrorMessage("Internal Fault")
                        .setErrorDetail("InternalFault: Unexpected error happened creating archive");
            }
            jobDAO.setJobError(job.getJobId(), jobException);
        }
    }

    private File getArchiveFile(ArchiveJob job) throws IOException {

        File parentDir = getArchiveParentDir(job.getPrincipal());

        if (!parentDir.exists()) {
            if (!parentDir.mkdirs()) {
                LOG.error("Unable to create directory " + parentDir.getAbsolutePath());
                throw new JobException(Type.FATAL).setErrorMessage("Internal Fault")
                        .setErrorDetail("InternalFault: Unable to create temporary directory for job");
            }
        }

        File archiveFile = parentDir.toPath().resolve(job.getJobId() + "." + job.getType().getExtension()).toFile();
        if (!archiveFile.createNewFile()) {
            LOG.error("Unable to create file " + archiveFile.getAbsolutePath());
            throw new JobException(Type.FATAL).setErrorMessage("Internal Fault")
                    .setErrorDetail("InternalFault: Unable to create archive file");
        }

        return archiveFile;
    }

    public File getArchiveParentDir(Principal principal) {
        return generatedDir.toPath().resolve(principal.getName()).toFile();
    }

    private String getCommonParent(List<String> vosPaths) {
        String commonParent = null;
        for (String vosPath : vosPaths) {
            if (commonParent == null) {
                commonParent = vosPath;
            } else {
                StringBuilder newCommonParent = new StringBuilder();
                boolean same = true;
                for (int i = 0; same && i < Math.min(commonParent.length(), vosPath.length()); i++) {
                    if (commonParent.charAt(i) == vosPath.charAt(i)) {
                        newCommonParent.append(commonParent.charAt(i));
                    } else {
                        same = false;
                    }
                }
                commonParent = newCommonParent.toString();
            }
        }
        return commonParent;
    }

    private static abstract class ArchiveHandler<O extends OutputStream, E> implements AutoCloseable {

        private final O os;

        ArchiveHandler(O os) {
            this.os = os;
        }

        public abstract E getEntry(File file, String path);

        public abstract void putNextEntry(E entry) throws IOException;

        public void putNextEntry(File file, String path) throws IOException {
            putNextEntry(getEntry(file, path));
        }

        public final O getOutputStream() {
            return os;
        }

        @Override
        public void close() throws Exception {
            os.close();
        }
    }

    private class TarArchiveHandler extends ArchiveHandler<TarOutputStream, TarEntry> {

        TarArchiveHandler(File archiveFile) throws IOException {
            super(new TarOutputStream(new BufferedOutputStream(new FileOutputStream(archiveFile))));
        }

        @Override
        public TarEntry getEntry(File file, String path) {
            return new TarEntry(file, path);
        }

        @Override
        public void putNextEntry(TarEntry entry) throws IOException {
            getOutputStream().putNextEntry(entry);
        }
    }

    private class ZipArchiveHandler extends ArchiveHandler<ZipOutputStream, ZipEntry> {

        ZipArchiveHandler(File archiveFile) throws IOException {
            super(new ZipOutputStream(new BufferedOutputStream(new FileOutputStream(archiveFile))));
        }

        @Override
        public ZipEntry getEntry(File file, String path) {
            if (file.isDirectory()) {
                // ZipEntry assumes that paths ending with / are folders
                path += "/";
            }
            return new ZipEntry(path);
        }

        @Override
        public void putNextEntry(ZipEntry entry) throws IOException {
            getOutputStream().putNextEntry(entry);
        }
    }

    private ArchiveHandler getArchiveHandler(File archiveFile, ArchiveJob.Type type) throws IOException {
        switch (type) {
            case TAR:
                return new TarArchiveHandler(archiveFile);
            case ZIP:
                return new ZipArchiveHandler(archiveFile);
            default:
                throw new UnsupportedOperationException("Type " + type + " not supported yet");
        }
    }

    private <O extends OutputStream, E> void downloadFileIntoArchive(FileInfo fileInfo, String relPath, TokenPrincipal tokenPrincipal, ArchiveHandler<O, E> handler, String baseUrl) {

        if (baseUrl == null) {
            LOG.error("Location URL not found for location " + fileInfo.getLocationId());
            throw new JobException(Type.FATAL).setErrorMessage("Internal Fault")
                    .setErrorDetail("InternalFault: Unable to retrieve location of file " + fileInfo.getVirtualPath());
        }

        String url = baseUrl + "/" + fileInfo.getVirtualName();

        LOG.trace("Downloading file from " + url);

        restTemplate.execute(url, HttpMethod.GET, req -> {
            HttpHeaders headers = req.getHeaders();
            if (tokenPrincipal.getToken() != null) {
                headers.setBearerAuth(tokenPrincipal.getToken());
            }
        }, res -> {
            File tmpFile = Files.createTempFile("download", null).toFile();
            try ( FileOutputStream os = new FileOutputStream(tmpFile)) {
                res.getBody().transferTo(os);
                handler.putNextEntry(tmpFile, relPath);
                try ( FileInputStream is = new FileInputStream(tmpFile)) {
                    is.transferTo(handler.getOutputStream());
                }
            } finally {
                tmpFile.delete();
            }
            return null;
        }, new Object[]{});
    }

    private <O extends OutputStream, E> void writeFileIntoArchive(FileInfo fileInfo, String relPath, TokenPrincipal tokenPrincipal, ArchiveHandler<O, E> handler) throws IOException {
        if (!authorizationService.isDownloadable(fileInfo, tokenPrincipal)) {
            throw new JobException(Type.FATAL).setErrorMessage("Permission Denied")
                    .setErrorDetail("PermissionDenied: " + fileInfo.getVirtualPath());
        }

        File file = new File(fileInfo.getOsPath());
        LOG.trace("Adding file " + file.getAbsolutePath() + " to tar archive");

        try ( InputStream is = new FileInputStream(file)) {
            handler.putNextEntry(file, relPath);
            is.transferTo(handler.getOutputStream());
        }
    }
}
