/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.controller;

import it.inaf.ia2.transfer.persistence.model.FileInfo;
import it.inaf.ia2.aa.jwt.TokenParser;
import it.inaf.ia2.transfer.auth.GmsClient;
import it.inaf.ia2.transfer.persistence.FileDAO;
import java.io.File;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import org.assertj.core.util.Files;
import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@AutoConfigureMockMvc
public class GetFileControllerTest {

    @MockBean
    private TokenParser tokenParser;

    @MockBean
    private GmsClient gmsClient;

    @MockBean
    private FileDAO fileDao;

    @Autowired
    private MockMvc mockMvc;

    private File tempFile;

    @BeforeEach
    public void init() {
        tempFile = Files.newTemporaryFile();
    }

    @AfterEach
    public void cleanup() {
        tempFile.delete();
    }

    @Test
    public void getPublicFile() throws Exception {

        FileInfo fileInfo = new FileInfo();
        fileInfo.setOsPath(tempFile.getAbsolutePath());
        fileInfo.setVirtualPath("/path/to/myfile");
        fileInfo.setPublic(true);

        when(fileDao.getFileInfo(eq("/path/to/myfile"))).thenReturn(Optional.of(fileInfo));

        mockMvc.perform(get("/path/to/myfile"))
                .andDo(print())
                .andExpect(status().isOk());
    }

    @Test
    public void testContextPathIsRemoved() throws Exception {

        FileInfo fileInfo = new FileInfo();
        fileInfo.setOsPath(tempFile.getAbsolutePath());
        fileInfo.setVirtualPath("/path/to/myfile");
        fileInfo.setPublic(true);

        when(fileDao.getFileInfo(eq("/path/to/myfile"))).thenReturn(Optional.of(fileInfo));

        mockMvc.perform(get("/context/path/to/myfile").contextPath("/context"))
                .andDo(print())
                .andExpect(status().isOk());
    }

    @Test
    public void testFileNotFoundInDb() throws Exception {
        mockMvc.perform(get("/path/to/myfile"))
                .andDo(print())
                .andExpect(status().isNotFound());
    }

    @Test
    public void testFileNotFoundOnDisk() throws Exception {

        FileInfo fileInfo = new FileInfo();
        fileInfo.setOsPath("/this/doesnt/exists");
        fileInfo.setPublic(true);

        when(fileDao.getFileInfo(any())).thenReturn(Optional.of(fileInfo));

        mockMvc.perform(get("/myfile"))
                .andDo(print())
                .andExpect(status().isNotFound());
    }

    @Test
    public void getPrivateFileTokenInHeader() throws Exception {

        prepareMocksForPrivateFile();

        mockMvc.perform(get("/path/to/myfile")
                .header("Authorization", "Bearer: <token>"))
                .andDo(print())
                .andExpect(status().isOk());

        verify(gmsClient).isMemberOf(eq("<token>"), eq("group1"));
    }

    @Test
    public void getPrivateFileTokenInQueryString() throws Exception {

        prepareMocksForPrivateFile();

        mockMvc.perform(get("/path/to/myfile?token=<token>"))
                .andDo(print())
                .andExpect(status().isOk());

        verify(gmsClient).isMemberOf(eq("<token>"), eq("group1"));
    }

    private void prepareMocksForPrivateFile() {

        Map<String, Object> claims = new HashMap<>();
        claims.put("sub", "123");

        when(tokenParser.getClaims(any())).thenReturn(claims);

        when(gmsClient.isMemberOf(any(), any())).thenReturn(true);

        FileInfo fileInfo = new FileInfo();
        fileInfo.setOsPath(tempFile.getAbsolutePath());
        fileInfo.setVirtualPath("/path/to/myfile");
        fileInfo.setGroupRead(Collections.singletonList("group1"));

        when(fileDao.getFileInfo(any())).thenReturn(Optional.of(fileInfo));
    }

    @Test
    public void getPrivateFileByOwnerId() throws Exception {

        Map<String, Object> claims = new HashMap<>();
        claims.put("sub", "123");

        when(tokenParser.getClaims(any())).thenReturn(claims);

        FileInfo fileInfo = new FileInfo();
        fileInfo.setVirtualPath("/path/to/myfile");
        fileInfo.setOsPath(tempFile.getAbsolutePath());
        fileInfo.setOwnerId("123");

        when(fileDao.getFileInfo(any())).thenReturn(Optional.of(fileInfo));

        mockMvc.perform(get("/path/to/myfile")
                .header("Authorization", "Bearer: <token>"))
                .andDo(print())
                .andExpect(status().isOk());
    }

    @Test
    public void testPrivateFileNullToken() throws Exception {

        FileInfo fileInfo = new FileInfo();
        fileInfo.setVirtualPath("/path/to/myfile");

        when(fileDao.getFileInfo(any())).thenReturn(Optional.of(fileInfo));

        mockMvc.perform(get("/path/to/myfile"))
                .andDo(print())
                .andExpect(status().isUnauthorized());
    }
}
