/*
 * This file is part of vospace-file-service
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.ia2.transfer.persistence;

import it.inaf.ia2.transfer.persistence.model.FileInfo;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import javax.sql.DataSource;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.junit.jupiter.SpringExtension;

@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = {DataSourceConfig.class})
@TestPropertySource(locations = "classpath:test.properties")
public class FileDAOTest {

    @Autowired
    private DataSource dataSource;

    private FileDAO dao;

    @BeforeEach
    public void init() {
        dao = new FileDAO(dataSource);
    }

    @Test
    public void testGetFileInfo() {

        Optional<FileInfo> optFileInfo = dao.getFileInfo("/test1/file1.txt");

        assertTrue(optFileInfo.isPresent());

        FileInfo fileInfo = optFileInfo.get();

        assertEquals("/home/username1/retrieve/file1.txt", fileInfo.getOsPath());
    }

    @Test
    public void testGetUnexistingFile() {
        assertTrue(dao.getFileInfo("/doesnt/exist").isEmpty());
    }

    @Test
    public void testGetArchiveFileInfos() {

        List<FileInfo> fileInfos = dao.getArchiveFileInfos(Arrays.asList("/public/file1", "/public/file2", "/public/subdir1"));

        assertEquals(6, fileInfos.size());

        assertEquals("/home/vospace/upload/user1/file1", fileInfos.get(0).getOsPath());
        assertEquals("/home/vospace/upload/user1/file2", fileInfos.get(1).getOsPath());
        assertTrue(fileInfos.get(2).isDirectory());
        assertEquals("/home/username1/retrieve/subdir1/file3", fileInfos.get(3).getOsPath());
        assertEquals("/home/username1/retrieve/subdir1/file4", fileInfos.get(4).getOsPath());
        assertEquals("portal-file", fileInfos.get(5).getVirtualName());
    }
}
