package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import net.ivoa.xml.vospace.v2.Node;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.beans.factory.annotation.Autowired;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.List;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.PutMapping;
import it.inaf.oats.vospace.exception.*;
import java.util.stream.Collectors;
import java.util.Arrays;

@RestController
public class CreateNodeController extends BaseNodeController {

    @Autowired
    private NodeDAO nodeDao;

    @Value("${vospace-authority}")
    private String authority;

    @PutMapping(value = {"/nodes", "/nodes/**"},
            consumes = {MediaType.APPLICATION_XML_VALUE, MediaType.APPLICATION_JSON_VALUE},
            produces = {MediaType.APPLICATION_XML_VALUE, MediaType.APPLICATION_JSON_VALUE})
    public Node createNode(@RequestBody Node node, User principal) {

        String path = getPath();

        List<String> userGroups = principal.getGroups();

        // Validate payload node URI
        if (!isValidURI(node.getUri())) {
            throw new InvalidURIException(node.getUri());
        }

        // Check if payload URI is consistent with http request
        if (!isUrlConsistentWithPayloadURI(node.getUri(), path)) {
            throw new InvalidURIException(node.getUri(), path);
        }

        // Check if another node is already present at specified path
        // This checks if the user is trying to insert the root node at "/" too
        if (nodeDao.listNode(path).isPresent()) {
            throw new DuplicateNodeException(path);
        }

        // Retrieve parent node 
        Node parentNode = nodeDao.listNode(getParentPath(path))
                .orElseThrow(() -> new ContainerNotFoundException(getParentPath(path)));

        // Check user write/ownership privilege against parent node
        List<String> groupWritePropValues
                = getNodePropertyByURI(parentNode, "ivo://ivoa.net/vospace/core#groupwrite");

        if (groupWritePropValues.isEmpty()) {
            throw new PermissionDeniedException(path);
        }

        List<String> nodeGroups
                = Arrays.asList(groupWritePropValues.get(0).split(" ", -1));

        if (userGroups == null || 
                !nodeGroups.stream().anyMatch((i) -> userGroups.contains(i))) {
            // If groups don't match check ownership at least
            List<String> nodeOwner
                    = getNodePropertyByURI(parentNode, "ivo://ivoa.net/vospace/core#creator");

            if (nodeOwner.isEmpty()
                    || !nodeOwner.get(0).equals(principal.getName())) {
                throw new PermissionDeniedException(path);
            }
        }

        // Check if parent node is not a Container node and in case throw
        // appropriate exception
        if (!parentNode.getType().equals("vos:ContainerNode")) {
            if (parentNode.getType().equals("vos:LinkNode")) {
                throw new LinkFoundException(getParentPath(path));
            } else {
                throw new ContainerNotFoundException(getParentPath(path));
            }
        }

        return node;
    }

    // Assuming that this service implementation uses only ! as a separator
    // in the authority part of the URI
    private boolean isValidURI(String nodeURI) {
        String parsedAuthority;
        if (!nodeURI.startsWith("vos://")) {
            return false;
        } else {
            parsedAuthority = nodeURI.replaceAll("vos://", "").split("/", -1)[0];
        }

        if (parsedAuthority.isEmpty()
                || !parsedAuthority.replace("~", "!").equals(authority)) {
            return false;
        }

        return true;
    }

    private boolean isUrlConsistentWithPayloadURI(String nodeURI, String path) {
        // It's assumed that nodeURI has been validated to be in the expected 
        // form vos://authority[!~]somepath/mynode..."

        return nodeURI.replaceAll("vos://[^/]+", "").equals(path);

    }

    private String getParentPath(String path) {
        String[] parsedPath = path.split("/");

        StringBuilder sb = new StringBuilder();

        for (int i = 0; i < parsedPath.length - 1; i++) {
            sb.append("/").append(parsedPath[i]);
        }

        return sb.toString();
    }

    // Returns all properties stored inside the node under the requested
    // property URI.    
    private List<String> getNodePropertyByURI(Node node, String propertyURI) {

        List<String> propertyList = node.getProperties().stream()
                .filter((i) -> i.getUri()
                .equals(propertyURI))
                .map((i) -> i.getValue())
                .collect(Collectors.toList());

        return propertyList;
    }
}
