package it.inaf.oats.vospace;

import static it.inaf.oats.vospace.VOSpaceXmlTestUtil.loadDocument;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.Optional;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import static org.junit.jupiter.api.Assertions.assertEquals;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;

@SpringBootTest
@AutoConfigureMockMvc
@ContextConfiguration(classes = {TokenFilterConfig.class})
@TestPropertySource(properties = "spring.main.allow-bean-definition-overriding=true")
public class ListNodeControllerTest {

    private static final String URI_PREFIX = "vos://example.com!vospace";

    @MockBean
    private NodeDAO dao;

    @Autowired
    private MockMvc mockMvc;    

    @Test
    public void testRootXml() throws Exception {

        when(dao.listNode(eq("/"))).thenReturn(getRootNode());

        String xml = mockMvc.perform(get("/nodes")
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().isOk())
                .andReturn().getResponse().getContentAsString();

        Document doc = loadDocument(xml);
        assertEquals("vos:node", doc.getDocumentElement().getNodeName());
        assertEquals("vos:ContainerNode", doc.getDocumentElement().getAttribute("xsi:type"));

        verify(dao, times(1)).listNode(eq("/"));
    }

    @Test
    public void testNodeXml() throws Exception {

        when(dao.listNode(eq("/mynode"))).thenReturn(Optional.of(getDataNode()));

        String xml = mockMvc.perform(get("/nodes/mynode")
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().isOk())
                .andReturn().getResponse().getContentAsString();

        Document doc = loadDocument(xml);
        assertEquals("vos:node", doc.getDocumentElement().getNodeName());
        assertEquals("vos:DataNode", doc.getDocumentElement().getAttribute("xsi:type"));

        verify(dao, times(1)).listNode(eq("/mynode"));
    }

    @Test
    public void testNodeNotFound() throws Exception {
        mockMvc.perform(get("/nodes/mynode")
                .accept(MediaType.APPLICATION_XML))
                .andExpect(status().isNotFound());
    }

    @Test
    public void testPermissionDeniedUser() throws Exception {
        Node node = getDataNodeByOwnership("user2", "group1");

        when(dao.listNode(eq("/mynode"))).thenReturn(Optional.of(node));

        mockMvc.perform(get("/nodes/mynode")
                .header("Authorization", "Bearer user1_token")
                .accept(MediaType.APPLICATION_XML))
                .andExpect(status().is4xxClientError());
    }

    @Test
    public void testGrantedByGroup() throws Exception {
        Node node = getDataNodeByOwnership("user1", "group1");

        when(dao.listNode(eq("/mynode"))).thenReturn(Optional.of(node));

        mockMvc.perform(get("/nodes/mynode")
                .header("Authorization", "Bearer user2_token")
                .accept(MediaType.APPLICATION_XML))
                .andExpect(status().is2xxSuccessful());
    }

    @Test
    public void testRemoveUnreadable() throws Exception {
        // Create container node
        ContainerNode root = (ContainerNode) getRootNode().get();

        Node node1 = getDataNodeByOwnership("user1", "group10");
        node1.setUri(URI_PREFIX + "/mynode1");
        root.getNodes().add(node1);

        Node node2 = getDataNodeByOwnership("user1", "group10");
        node2.setUri(URI_PREFIX + "/mynode2");
        root.getNodes().add(node2);

        Node node3 = getDataNodeByOwnership("user2", "group10");
        node3.setUri(URI_PREFIX + "/mynode3");
        root.getNodes().add(node3);

        Node node4 = getDataNodeByOwnership("user3", "group10");
        node4.setUri(URI_PREFIX + "/mynode4");
        root.getNodes().add(node4);

        when(dao.listNode(eq("/"))).thenReturn(Optional.of(root));

        String xml = mockMvc.perform(get("/nodes/")
                .header("Authorization", "Bearer user2_token")
                .accept(MediaType.APPLICATION_XML))
                .andExpect(status().is2xxSuccessful())
                .andDo(print())
                .andReturn().getResponse().getContentAsString();
        
        Document doc = loadDocument(xml);
        assertEquals("vos:node", doc.getDocumentElement().getNodeName());
        assertEquals("vos:ContainerNode", doc.getDocumentElement().getAttribute("xsi:type"));
        NodeList nl = doc.getDocumentElement().getElementsByTagName("vos:nodes");        
        
        assertEquals(1, nl.getLength());
        NodeList children = nl.item(0).getChildNodes();
        assertEquals(2, children.getLength());
        verify(dao, times(1)).listNode(eq("/"));

    }

    private Optional<Node> getRootNode() {
        ContainerNode root = new ContainerNode();
        root.setUri(URI_PREFIX + "/");
        Property publicProperty = new Property();
        publicProperty.setUri(NodeProperties.PUBLIC_READ_URI);
        publicProperty.setValue("true");
        root.getProperties().add(publicProperty);

        root.getNodes().add(getDataNode());
        return Optional.of(root);
    }

    private Node getDataNode() {
        DataNode node = new DataNode();
        node.setUri(URI_PREFIX + "/mynode");
        Property publicProperty = new Property();
        publicProperty.setUri(NodeProperties.PUBLIC_READ_URI);
        publicProperty.setValue("true");
        node.getProperties().add(publicProperty);

        return node;
    }

    private Node getDataNodeByOwnership(String ownerID, String group) {
        DataNode node = new DataNode();
        node.setUri(URI_PREFIX + "/mynode");
        // Set owner
        Property creatorProperty = new Property();
        creatorProperty.setUri(NodeProperties.CREATOR_URI);
        creatorProperty.setValue(ownerID);
        node.getProperties().add(creatorProperty);
        // set group  
        Property readGroupProperty = new Property();
        readGroupProperty.setUri(NodeProperties.GROUP_READ_URI);
        readGroupProperty.setValue(group);
        node.getProperties().add(readGroupProperty);

        return node;
    }
}
