package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.persistence.JobDAO;
import java.util.Optional;
import java.util.UUID;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.vospace.v2.Transfer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PathVariable;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

@RestController
public class TransferController {

    @Autowired
    private JobDAO jobDAO;

    @Autowired
    private HttpServletRequest request;

    @Autowired
    private JobService jobService;

    @PostMapping(value = "/transfers",
            consumes = {MediaType.APPLICATION_XML_VALUE, MediaType.TEXT_XML_VALUE, MediaType.APPLICATION_JSON_VALUE},
            produces = {MediaType.APPLICATION_XML_VALUE, MediaType.TEXT_XML_VALUE, MediaType.APPLICATION_JSON_VALUE})
    public ResponseEntity<?> postTransfer(@RequestBody Transfer transfer,
            @RequestParam(value = "PHASE", required = false) Optional<String> phase, User principal) {

        String jobId = UUID.randomUUID().toString().replace("-", "");

        JobSummary jobSummary = new JobSummary();
        jobSummary.setJobId(jobId);
        jobSummary.setOwnerId(principal.getName());
        jobSummary.setPhase(ExecutionPhase.PENDING);
        JobSummary.JobInfo jobInfo = new JobSummary.JobInfo();
        jobInfo.getAny().add(transfer);
        jobSummary.setJobInfo(jobInfo);

        jobDAO.createJob(jobSummary);

        if (phase.isPresent() && "RUN".equals(phase.get())) {
            jobService.startJob(jobSummary);
        }

        return getJobRedirect(jobId);
    }

    @GetMapping(value = "/transfers/{jobId}", produces = {MediaType.APPLICATION_XML_VALUE, MediaType.TEXT_XML_VALUE, MediaType.APPLICATION_JSON_VALUE})
    public ResponseEntity<JobSummary> getJob(@PathVariable("jobId") String jobId) {
        return jobDAO.getJob(jobId).map(j -> ResponseEntity.ok(j)).orElse(ResponseEntity.notFound().build());
    }

    @PostMapping(value = "/transfers/{jobId}/phase")
    public ResponseEntity<?> setJobPhase(@PathVariable("jobId") String jobId, @RequestParam("PHASE") String phase, User principal) {

        return jobDAO.getJob(jobId).map(job -> {
            if (!job.getOwnerId().equals(principal.getName())) {
                return ResponseEntity.status(HttpStatus.UNAUTHORIZED).build();
            }

            // TODO: check allowed job phase transitions
            switch (phase) {
                case "RUN":
                    jobService.startJob(job);
                    break;
                case "ABORT":
                    throw new UnsupportedOperationException("Not implemented yet");
                default:
                    throw new IllegalArgumentException("Invalid phase parameter: " + phase);
            }

            return getJobRedirect(job.getJobId());

        }).orElse(ResponseEntity.notFound().build());
    }

    private ResponseEntity<?> getJobRedirect(String jobId) {
        HttpHeaders headers = new HttpHeaders();
        headers.set("Location", request.getContextPath() + "/transfers/" + jobId);
        return new ResponseEntity<>(headers, HttpStatus.SEE_OTHER);
    }
}
