package it.inaf.oats.vospace.persistence;

import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import java.util.ArrayList;
import java.util.List;
import javax.sql.DataSource;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.View;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.junit.jupiter.SpringExtension;

@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = {DataSourceConfig.class})
@TestPropertySource(locations = "classpath:test.properties")
public class NodeDAOTest {

    @Autowired
    private DataSource dataSource;
    private NodeDAO dao;

    @BeforeEach
    public void init() {
        dao = new NodeDAO(dataSource);
    }

    @Test
    public void testCreateNode() {
        DataNode dataNode = new DataNode();

        dataNode.setUri("vos://example.com!vospace/mydata1");
        dataNode.setAccepts(getViews());
        dataNode.setProvides(getViews());
        dataNode.setBusy(true);

        dao.createNode(dataNode);

        DataNode retrievedNode = (DataNode) dao.listNode("/mydata1").get();

        assertEquals(retrievedNode.getAccepts().get(0).getUri(), dataNode.getAccepts().get(0).getUri());
        assertEquals(retrievedNode.getProvides().get(0).getUri(), dataNode.getProvides().get(0).getUri());
    }

    @Test
    public void testListNode() {
        ContainerNode root = (ContainerNode) dao.listNode("/").get();
        assertEquals(2, root.getNodes().size());
        
        assertEquals("true", NodeProperties.getNodePropertyAsListByURI(root, NodeProperties.PUBLIC_READ_URI).get(0));

        assertEquals("group1 group2", getProperty(root.getNodes().get(0), "ivo://ivoa.net/vospace/core#groupread"));
    }

    @Test
    public void testCountNodeWithPath() {
        assertEquals(1, dao.countNodesWithPath("/"));
        assertEquals(1, dao.countNodesWithPath("/test1"), "Test db has been changed");
        assertEquals(1, dao.countNodesWithPath("/test1/f1"), "Test db has been changed");
        assertEquals(1, dao.countNodesWithPath("/test1/f1/f2_renamed"), "Test db has been changed");
        assertEquals(1, dao.countNodesWithPath("/test1/f1/f2_renamed/f3"), "Test db has been changed");

        assertEquals(1, dao.countNodesWithPath("/test2"), "Test db has been changed");

        assertEquals(1, dao.countNodesWithPath("/test2/f4"), "Test db has been changed");
        assertEquals(1, dao.countNodesWithPath("/test2/f5"), "Test db has been changed");

        assertEquals(0, dao.countNodesWithPath("/pippooo"), "Test db has been changed");
    }

    @Test
    public void testDeleteNode() {

        assertEquals(1, dao.countNodesWithPath("/test1/f1/f2_renamed/f3"), "Test db has been changed");
        dao.deleteNode("/test1");

        assertEquals(0, dao.countNodesWithPath("/test1"));
        assertEquals(0, dao.countNodesWithPath("/test1/f1"));
        assertEquals(0, dao.countNodesWithPath("/test1/f1/f2_renamed"));
        assertEquals(0, dao.countNodesWithPath("/test1/f1/f2_renamed/f3"));

    }

    private String getProperty(Node node, String uri) {
        for (Property property : node.getProperties()) {
            if (uri.equals(property.getUri())) {
                return property.getValue();
            }
        }
        return null;
    }

    private List<View> getViews() {
        View view = new View();
        view.setUri("urn:myview");
        List<View> views = new ArrayList<>();
        views.add(view);
        return views;
    }
}
