/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace.persistence;

import it.inaf.oats.vospace.persistence.model.NodeCollection;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.List;
import java.util.Optional;
import javax.sql.DataSource;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.stereotype.Repository;

/**
 *
 * @author Nicola Fulvio Calabria <nicola.calabria at inaf.it>
 */
@Repository
public class CollectionsDAO {

    private final JdbcTemplate jdbcTemplate;

    @Autowired
    public CollectionsDAO(DataSource dataSource) {
        jdbcTemplate = new JdbcTemplate(dataSource);
    }

    void createNewCollection(String title, String ownerId) {

        String sql = "INSERT INTO collections (title, owner_id) VALUES (?,?)";

        jdbcTemplate.update(conn -> {
            PreparedStatement ps = conn.prepareStatement(sql);
            int i = 0;
            ps.setString(++i, title);
            ps.setString(++i, ownerId);
            return ps;
        });
    }

    Optional<NodeCollection> getNodeCollectionById(Long id) {
        String sql = "SELECT collection_id, title, owner_id FROM collections\n"
                + "WHERE collection_id = ?";

        Optional<NodeCollection> nc = jdbcTemplate.query(
                conn -> {
                    PreparedStatement ps = conn.prepareStatement(sql);
                    ps.setLong(1, id);
                    return ps;
                },
                rs -> {
                    if (!rs.next()) {
                        return Optional.empty();
                    } else {
                        return Optional.of(getNodeCollectionFromResultset(rs));
                    }

                });

        return nc;

    }
    
    List<NodeCollection> getUserNodeCollections(String userId) {
        String sql = "SELECT collection_id, title, owner_id FROM collections\n"
                + "WHERE owner_id = ?";

        List<NodeCollection> nc = jdbcTemplate.query(
                conn -> {
                    PreparedStatement ps = conn.prepareStatement(sql);
                    ps.setString(1, userId);
                    return ps;
                },
                (row, index) -> {
                    return this.getNodeCollectionFromResultset(row);
                });

        return nc;
    }

    private NodeCollection getNodeCollectionFromResultset(ResultSet rs)
            throws SQLException {
        NodeCollection nc = new NodeCollection(
                rs.getLong("collection_id"),
                rs.getString("title"),
                rs.getString("owner_id")
        );

        return nc;
    }
}
