package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import it.inaf.oats.vospace.exception.NodeBusyException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.List;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Transfer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class MoveService {

    @Autowired
    private NodeDAO nodeDao;

    @Autowired
    private HttpServletRequest servletRequest;

    public void processMoveJob(JobSummary job, Transfer transfer) {

        // Get Source Path
        String sourcePath = transfer.getTarget();

        // Get Destination Path (it's in transfer direction)
        String destinationPath = transfer.getDirection();

        // Extract User permissions from servlet request
        User user = (User) servletRequest.getUserPrincipal();

        Long sourceId = nodeDao.getNodeId(sourcePath);
        List<Node> branchList = nodeDao.listNodesInBranch(sourceId, true);
        
        // Check feasibility of move on source branch
        if (!isWritePermissionsValid(branchList, user)) {
            throw new PermissionDeniedException(sourcePath);
        }
        
        if(sourcePath.equals(destinationPath))
            return;
        
        if(!isMoveable(branchList)) {
            throw new NodeBusyException(sourcePath);
        }
        
        // Set branch at busy        
        nodeDao.setBranchBusy(sourceId, true);
               
        
        // Compare source and destination paths and see if it's just a rename
        if(NodeUtils.getParentPath(sourcePath).equals(NodeUtils.getParentPath(destinationPath)))
        {           
            nodeDao.renameNode(sourceId, NodeUtils.getLastPathElement(destinationPath));
        } else {
            this.moveNode(sourceId, sourcePath, destinationPath, user);
        }

        nodeDao.setBranchBusy(sourceId, false);
        
    }

    // All nodes must be writable by the user to have a true
    private boolean isWritePermissionsValid(List<Node> list, User user) {
        String userName = user.getName();
        List<String> userGroups = user.getGroups();

        return list.stream().allMatch((n) -> {
            return NodeUtils.checkIfWritable(n, userName, userGroups);
        });

    }

    // All nodes must comply to have a true            
    private boolean isMoveable(List<Node> list) {
        return list.stream().allMatch((n) -> {
            boolean busy = NodeUtils.getIsBusy(n);
            boolean sticky
                    = Boolean.valueOf(
                            NodeProperties.getNodePropertyByURI(n,
                                    NodeProperties.STICKY_URN));

            return (!busy && !sticky);
        });
    }
       
    
    private void moveNode(Long sourceId, String sourcePath, String destPath, User user)
    {
        
    }

}
