/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.exception.NodeBusyException;
import it.inaf.oats.vospace.exception.NodeNotFoundException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import it.inaf.oats.vospace.persistence.DataSourceConfigSingleton;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.List;
import java.util.Optional;
import net.ivoa.xml.vospace.v2.Transfer;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.MethodOrderer.OrderAnnotation;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.ContextConfiguration;

@SpringBootTest
@AutoConfigureMockMvc
@ContextConfiguration(classes = DataSourceConfigSingleton.class)
@TestPropertySource(locations = "classpath:test.properties", properties = {"vospace-authority=example.com!vospace", "file-service-url=http://file-service"})
@TestMethodOrder(OrderAnnotation.class)
@DirtiesContext(classMode = DirtiesContext.ClassMode.AFTER_EACH_TEST_METHOD)
public class CopyServiceTest {

    @Value("${vospace-authority}")
    private String authority;

    @Autowired
    private CopyService copyService;

    @Autowired
    private NodeDAO nodeDao;

    @Test
    @Order(1)
    public void copyRootTest() {

        assertThrows(IllegalArgumentException.class, () -> {
            copyService.processCopyNodes(getTransfer("/", "/pippo"), "job_pippo", getAnonymousUser());
        }
        );

        assertThrows(IllegalArgumentException.class, () -> {
            copyService.processCopyNodes(getTransfer("/pippo", "/"), "job_pippo", getAnonymousUser());
        }
        );

    }

    @Test
    @Order(2)
    public void testNonExistingSourceNode() {
        assertThrows(NodeNotFoundException.class, () -> {
            copyService.processCopyNodes(getTransfer("/pippo", "/test2"), "job_pippo", getAnonymousUser());
        }
        );
    }

    @Test
    @Order(3)
    public void testCopyDeniedOnBusySource() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        assertThrows(NodeBusyException.class, () -> {
            copyService.processCopyNodes(getTransfer("/test3/mbusy", "/test3/m1"), "job_pippo", user);
        }
        );
    }

    @Test
    @Order(4)
    public void testPermissionDeniedOnSource() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user1");

        assertThrows(PermissionDeniedException.class, () -> {
            copyService.processCopyNodes(getTransfer("/test3/m1", "/test4"), "job_pippo", user);
        }
        );
    }

    @Test
    @Order(5)
    public void testPermissionDeniedOnExistingDestination() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user1");
        when(user.getGroups()).thenReturn(List.of("group1"));

        assertThrows(PermissionDeniedException.class, () -> {
            copyService.processCopyNodes(getTransfer("/test3/group1", "/test3/m1/m2"), "job_pippo", user);
        }
        );
    }

    @Test
    @Order(6)
    public void testDestinationExistsAndIsBusy() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        assertThrows(NodeBusyException.class, () -> {
            copyService.processCopyNodes(getTransfer("/test3/m1", "/test3/mbusy"), "job_pippo", user);
        }
        );
    }

    @Test
    @Order(7)
    public void testCopyToExistingDestination() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        // Preliminary checks for assumptions
        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());

        Optional<Long> destId = nodeDao.getNodeId("/test4");
        assertTrue(destId.isPresent());

        // copy
        String copyDestination
                = copyService.processCopyNodes(getTransfer("/test3/m1", "/test4"), "job_pippo", user).get(1);

        assertEquals("/test4/m1", copyDestination);

        // source has been moved
        Optional<Long> oldSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(oldSourceId.isPresent());
        Optional<Long> oldChildId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(oldChildId.isPresent());

        Optional<Long> newSourceId = nodeDao.getNodeId("/test4/m1");
        assertTrue(newSourceId.isPresent());

        Optional<Long> newChildId = nodeDao.getNodeId("/test4/m1/m2");
        assertTrue(newChildId.isPresent());

    }

    @Test
    @Order(8)
    public void testCopyToExistingParent() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        // Preliminary checks for assumptions
        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());

        Optional<Long> destId = nodeDao.getNodeId("/test3/m1/m2_copy");
        assertTrue(destId.isEmpty());

        // copy
        String copyDestination
                = copyService.processCopyNodes(getTransfer("/test3/m1/m2", "/test3/m1/m2_copy"), "job_pippo", user).get(1);

        assertEquals("/test3/m1/m2_copy", copyDestination);

        // source has been moved
        Optional<Long> oldSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(oldSourceId.isPresent());
        Optional<Long> oldChildId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(oldChildId.isPresent());

        Optional<Long> newSourceId = nodeDao.getNodeId("/test3/m1/m2_copy");
        assertTrue(newSourceId.isPresent());

    }

    @Test
    @Order(9)
    public void testCopyDeniedToExistingDestination() {

        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        // Preliminary checks for assumptions
        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());

        assertThrows(IllegalArgumentException.class, () -> {
            copyService.processCopyNodes(getTransfer("/test3/m1/m2", "/test3/m1"), "job_pippo", user);
        }
        );
    }

    private Transfer getTransfer(String vosTarget, String vosDestination) {
        Transfer transfer = new Transfer();
        transfer.setTarget("vos://" + this.authority + vosTarget);
        transfer.setDirection("vos://" + this.authority + vosDestination);
        return transfer;
    }

    private User getAnonymousUser() {
        return new User().setUserId("anonymous");
    }
}
