/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.exception.InternalFaultException;
import it.inaf.oats.vospace.exception.NodeBusyException;
import it.inaf.oats.vospace.exception.NodeNotFoundException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import it.inaf.oats.vospace.persistence.DataSourceConfigSingleton;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.Optional;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.Test;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.annotation.DirtiesContext;

@SpringBootTest
@AutoConfigureMockMvc
@ContextConfiguration(classes = DataSourceConfigSingleton.class)
@TestPropertySource(locations = "classpath:test.properties", properties = {"vospace-authority=example.com!vospace", "file-service-url=http://file-service"})
@DirtiesContext(classMode = DirtiesContext.ClassMode.AFTER_EACH_TEST_METHOD)
public class DeleteNodeServiceTest {    
 
    @Autowired
    private DeleteNodeService deleteNodeService;
    
    @Autowired
    private NodeDAO nodeDao;
    
    
    @Test
    public void testPreliminaryCheckOnRoot() {
        assertThrows(PermissionDeniedException.class, ()->{
            deleteNodeService.doPreliminaryChecks("/");
        });
    }
    
    @Test
    public void testPreliminaryNodeNotFound() {
        assertThrows(NodeNotFoundException.class, ()->{
            deleteNodeService.doPreliminaryChecks("/nonexistent/node");
        });
    }
    
    @Test
    public void deleteBranchTest() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());
        // Delete
        deleteNodeService.deleteNode("/test3/m1", user);

        Optional<Long> checkSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(checkSourceId.isEmpty());
        
        Optional<Long> checkSourceIdChild = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(checkSourceIdChild.isEmpty());        
        
    }
    
    @Test
    public void deleteStopOnImmutableTest() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());
        
        nodeDao.setBranchImmutable(childId.get(), true);
        
        // Delete
        assertThrows(InternalFaultException.class, 
                ()->{ deleteNodeService.deleteNode("/test3/m1", user);}
        );

        Optional<Long> checkSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(checkSourceId.isPresent());
        
        Optional<Long> checkSourceIdChild = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(checkSourceIdChild.isPresent());        
    }
    
    @Test
    public void deleteStopOnBusyTest() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user3");

        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());
        
        nodeDao.setBranchJobId(childId.get(), "pippo");
        
        // Delete
        assertThrows(NodeBusyException.class, 
                ()->{ deleteNodeService.deleteNode("/test3/m1", user);}
        );

        Optional<Long> checkSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(checkSourceId.isPresent());
        
        Optional<Long> checkSourceIdChild = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(checkSourceIdChild.isPresent());        
    }
    
    
    @Test
    public void deleteStopOnPermissionDeniedTest() {
        User user = mock(User.class);
        when(user.getName()).thenReturn("user-pippo");

        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());
      
        // Delete
        assertThrows(PermissionDeniedException.class, 
                ()->{ deleteNodeService.deleteNode("/test3/m1", user);}
        );

        Optional<Long> checkSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(checkSourceId.isPresent());
        
        Optional<Long> checkSourceIdChild = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(checkSourceIdChild.isPresent());        
    }

}
