/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import net.ivoa.xml.vospace.v2.Node;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.beans.factory.annotation.Autowired;
import it.inaf.oats.vospace.persistence.NodeDAO;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.PutMapping;
import it.inaf.oats.vospace.exception.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import net.ivoa.xml.vospace.v2.Property;

@RestController
public class CreateNodeController extends BaseNodeController {

    private static final Logger LOG = LoggerFactory.getLogger(CreateNodeController.class);

    @Autowired
    private NodeDAO nodeDao;

    @Value("${vospace-authority}")
    private String authority;

    @PutMapping(value = {"/nodes", "/nodes/**"},
            consumes = {MediaType.APPLICATION_XML_VALUE, MediaType.TEXT_XML_VALUE, MediaType.APPLICATION_JSON_VALUE},
            produces = {MediaType.APPLICATION_XML_VALUE, MediaType.TEXT_XML_VALUE, MediaType.APPLICATION_JSON_VALUE})
    public Node createNode(@RequestBody Node node, User principal) {
        
        LOG.debug("createNode called for node with URI {}", node.getUri());
        
        // Validate payload node URI
        if (!isValidURI(node.getUri())) {
            throw new InvalidURIException(node.getUri());
        }        
        
        String path;        
        
        if(getPath() == null) {
            LOG.debug("createNode called internally with null path");
            path = node.getUri().replaceAll("vos://[^/]+", "");
        } else {            
            path = getPath();
            LOG.debug("createNode called for path {}", path);
        }

        // Check if payload URI is consistent with http request
        if (!isUrlConsistentWithPayloadURI(node.getUri(), path)) {
            throw new InvalidURIException(node.getUri(), path);
        }

        // Check if another node is already present at specified path
        // This checks if the user is trying to insert the root node at "/" too
        if (nodeDao.listNode(path).isPresent()) {
            throw new DuplicateNodeException(path);
        }

        // Retrieve parent node 
        Node parentNode = nodeDao.listNode(getParentPath(path))
                .orElseThrow(() -> new ContainerNotFoundException(getParentPath(path)));

        // Check if parent node is not a Container node and in case throw
        // appropriate exception
        if (!parentNode.getType().equals("vos:ContainerNode")) {
            if (parentNode.getType().equals("vos:LinkNode")) {
                throw new LinkFoundException(getParentPath(path));
            } else {
                throw new ContainerNotFoundException(getParentPath(path));
            }
        }

        if (!NodeUtils.checkIfWritable(parentNode, principal.getName(), principal.getGroups())) {
            throw new PermissionDeniedException(path);
        }

        // Check if node creator property is set. If not set it according to 
        // token. In case of creator mistmatch between node and token throw
        // exception
        String creator = NodeProperties.getNodePropertyByURI(
                node, NodeProperties.CREATOR_URI);

        if (creator == null) {
            Property creatorProperty = new Property();
            creatorProperty.setUri(NodeProperties.CREATOR_URI);
            creatorProperty.setValue(principal.getName());
            node.getProperties().add(creatorProperty);
        } else {
            if (!creator.equals(principal.getName())) // maybe a more specific exception would be more appropriate?
            {
                throw new PermissionDeniedException(path);
            }
        }

        // If payload has no read groups specified, inherit from parent node
        String payloadReadGroups = NodeProperties.getNodePropertyByURI(
                node, NodeProperties.GROUP_READ_URI);

        if (payloadReadGroups == null) {
            String parentNodeReadGroups = NodeProperties.getNodePropertyByURI(
                    parentNode, NodeProperties.GROUP_READ_URI);
            if (parentNodeReadGroups != null) {
                Property readGroups = new Property();
                readGroups.setUri(NodeProperties.GROUP_READ_URI);
                readGroups.setValue(parentNodeReadGroups);
                node.getProperties().add(readGroups);
            }
        }

        // If payload has no write groups specified, inherit from parent node
        String payloadWriteGroups = NodeProperties.getNodePropertyByURI(
                node, NodeProperties.GROUP_WRITE_URI);

        if (payloadWriteGroups == null) {
            String parentNodeWriteGroups = NodeProperties.getNodePropertyByURI(
                    parentNode, NodeProperties.GROUP_WRITE_URI);
            if (parentNodeWriteGroups != null) {
                Property writeGroups = new Property();
                writeGroups.setUri(NodeProperties.GROUP_WRITE_URI);
                writeGroups.setValue(parentNodeWriteGroups);
                node.getProperties().add(writeGroups);
            }
        }

        nodeDao.createNode(node);

        return node;
    }

    // Assuming that this service implementation uses only ! as a separator
    // in the authority part of the URI
    private boolean isValidURI(String nodeURI) {
        String parsedAuthority;
        if (!nodeURI.startsWith("vos://")) {
            return false;
        } else {
            parsedAuthority = nodeURI.replaceAll("vos://", "").split("/", -1)[0];
        }

        if (parsedAuthority.isEmpty()
                || !parsedAuthority.replace("~", "!").equals(authority)) {
            return false;
        }

        return true;
    }

    private boolean isUrlConsistentWithPayloadURI(String nodeURI, String path) {
        // It's assumed that nodeURI has been validated to be in the expected 
        // form vos://authority[!~]somepath/mynode..."

        return nodeURI.replaceAll("vos://[^/]+", "").equals(path);
    }
}
