/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import java.util.ArrayList;
import java.util.List;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.vospace.v2.Protocol;
import net.ivoa.xml.vospace.v2.Transfer;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.fail;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.anyString;
import org.mockito.MockedConstruction;
import org.mockito.Mockito;
import static org.mockito.Mockito.doAnswer;
import org.mockito.junit.jupiter.MockitoExtension;
import redis.clients.jedis.Jedis;

@ExtendWith(MockitoExtension.class)
public class AsyncTransferServiceTest {

    private static final String JSON_JOB = "{\"jobId\":\"job_id\",\"runId\":null,\"ownerId\":null,\"phase\":null,\"quote\":null,\"creationTime\":null,\"startTime\":null,\"endTime\":null,\"executionDuration\":0,\"destruction\":null,\"parameters\":null,\"results\":[],\"errorSummary\":null,\"jobInfo\":{\"transfer\":{\"target\":\"vos://example.com!vospace/my-node\",\"direction\":\"pullToVoSpace\",\"view\":null,\"protocols\":[{\"endpoint\":null,\"param\":[],\"uri\":\"ia2:async-recall\"}],\"keepBytes\":null,\"version\":null,\"param\":[]}},\"version\":null}\n";

    @Test
    public void testRedisRpc() {
        try (MockedConstruction<Jedis> staticMock = Mockito.mockConstruction(Jedis.class,
                (mockedJedis, context) -> {
                    doAnswer(invocation -> {
                        String requestId = invocation.getArgument(1);
                        List<String> result = new ArrayList<>();
                        result.add(requestId);
                        result.add(JSON_JOB);
                        return result;
                    }).when(mockedJedis).brpop(anyInt(), anyString());
                })) {
            AsyncTransferService asyncTransferService = new AsyncTransferService("localhost", 6379);
            JobSummary result = asyncTransferService.startJob(getFakeJob());
            assertEquals("job_id", result.getJobId());
        }
    }

    @Test
    public void testRedisError() {
        try (MockedConstruction<Jedis> staticMock = Mockito.mockConstruction(Jedis.class)) {
            AsyncTransferService asyncTransferService = new AsyncTransferService("localhost", 6379);
            try {
                asyncTransferService.startJob(getFakeJob());
                fail();
            } catch(IllegalStateException ex) {
            }
        }
    }
    
    private JobSummary getFakeJob() {
        Transfer transfer = new Transfer();
        transfer.setDirection("pullToVoSpace");
        Protocol protocol = new Protocol();
        protocol.setUri("ia2:async-recall");
        transfer.getProtocols().add(protocol);
        transfer.setTarget("vos://example.com!vospace/my-node");

        JobSummary job = new JobSummary();
        job.setJobId("job_id");

        JobSummary.JobInfo jobInfo = new JobSummary.JobInfo();
        jobInfo.getAny().add(transfer);

        job.setJobInfo(jobInfo);

        return job;
    }
}
