/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.exception.DuplicateNodeException;
import it.inaf.oats.vospace.exception.InternalFaultException;
import it.inaf.oats.vospace.exception.NodeBusyException;
import it.inaf.oats.vospace.exception.NodeNotFoundException;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import it.inaf.oats.vospace.persistence.DataSourceConfigSingleton;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.Optional;
import javax.servlet.http.HttpServletRequest;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.Transfer;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.MethodOrderer.OrderAnnotation;
import org.junit.jupiter.api.Order;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.TestMethodOrder;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.ContextConfiguration;

@SpringBootTest
@AutoConfigureMockMvc
@ContextConfiguration(classes = {DataSourceConfigSingleton.class, MoveServiceTest.TestConfig.class})
@TestPropertySource(locations = "classpath:test.properties", properties = {"vospace-authority=example.com!vospace", "file-service-url=http://file-service"})
@TestMethodOrder(OrderAnnotation.class)
@DirtiesContext(classMode = DirtiesContext.ClassMode.AFTER_EACH_TEST_METHOD)
public class MoveServiceTest {

    @Value("${vospace-authority}")
    private String authority;
    
    @Autowired
    private MoveService moveService;
    
    @Autowired
    private NodeDAO nodeDao;

    @Autowired
    private HttpServletRequest servletRequest;
       
    @TestConfiguration
    public static class TestConfig {

        /**
         * Necessary because MockBean doesn't work with HttpServletRequest.
         */
        @Bean
        @Primary
        public HttpServletRequest servletRequest() {
            HttpServletRequest request = mock(HttpServletRequest.class);
            User user = new User().setUserId("anonymous");
            when(request.getUserPrincipal()).thenReturn(user);
            return request;
        }
    }
    
    @Test
    @Order(1)
    public void moveRootTest() {

        assertThrows(IllegalArgumentException.class, () -> {
            moveService.processMoveJob(getTransfer("/", "/pippo"));
        }
        );

        assertThrows(IllegalArgumentException.class, () -> {
            moveService.processMoveJob(getTransfer("/pippo", "/"));
        }
        );

    }
    
    @Test
    @Order(2)
    public void testNonExistingSourceNode() {        
        assertThrows(NodeNotFoundException.class, () -> {
            moveService.processMoveJob(getTransfer("/pippo", "/test2"));
        }
        );        
    }  
    
    @Test
    @Order(3)
    public void testMoveDeniedOnAsync() {        
        assertThrows(InternalFaultException.class, () -> {
            moveService.processMoveJob(getTransfer("/test1", "/test4"));
        }
        );        
    }
    
    @Test
    @Order(4)
    public void testPermissionDenied() {
        User user = mock(User.class);        
        when(user.getName()).thenReturn("user1");
        when(servletRequest.getUserPrincipal()).thenReturn(user);
        
        assertThrows(PermissionDeniedException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/m1", "/test4"));
        }
        );        
    }    
    
    @Test
    @Order(5)
    public void testDestinationNodeAlreadyExisting() {
        User user = mock(User.class);        
        when(user.getName()).thenReturn("user3");
        when(servletRequest.getUserPrincipal()).thenReturn(user);        
        
        assertThrows(DuplicateNodeException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/m1", "/test4"));
        }
        );        
    }
    
    @Test
    @Order(6)
    public void testBusyNodeInSourceBranch() {
        User user = mock(User.class);        
        when(user.getName()).thenReturn("user3");
        when(servletRequest.getUserPrincipal()).thenReturn(user);
        
        nodeDao.setBranchBusy(nodeDao.getNodeId("/test3/m1/m2").orElseThrow(), true);
        
        assertThrows(NodeBusyException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/m1", "/test4"));
        }
        );
                
        nodeDao.setBranchBusy(nodeDao.getNodeId("/test3/m1/m2").orElseThrow(), false);        
    }
    
    @Test
    @Order(7)
    public void testNoMoveOnSticky() {
        User user = mock(User.class);        
        when(user.getName()).thenReturn("user3");
        when(servletRequest.getUserPrincipal()).thenReturn(user);
        
        assertThrows(NodeBusyException.class, () -> {
            moveService.processMoveJob(getTransfer("/test3/mstick", "/test4"));
        }
        );
        
    }
    
    @Test
    @Order(8)
    public void testRenameNode() {
        User user = mock(User.class);        
        when(user.getName()).thenReturn("user3");
        when(servletRequest.getUserPrincipal()).thenReturn(user);
        
        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());
        // Rename
        moveService.processMoveJob(getTransfer("/test3/m1", "/test3/m1ren"));
        
        Optional<Long> checkSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(checkSourceId.isEmpty());
        
        Optional<Long> newSourceId = nodeDao.getNodeId("/test3/m1ren");
        assertTrue(newSourceId.isPresent());
        assertEquals(sourceId.get(), newSourceId.get());
        
        Optional<Long> newChildId = nodeDao.getNodeId("/test3/m1ren/m2");
        assertTrue(newChildId.isPresent());
        assertEquals(childId.get(), newChildId.get());        
        
    }
    
    @Test
    @Order(9)
    public void testMoveToExistingParent(){
        User user = mock(User.class);        
        when(user.getName()).thenReturn("user3");
        when(servletRequest.getUserPrincipal()).thenReturn(user);
        
        // Preliminary checks for assumptions
        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());
        
        Optional<Long> destParentId = nodeDao.getNodeId("/test4");
        assertTrue(destParentId.isPresent());
        
        Optional<Long> destId = nodeDao.getNodeId("/test4/dest1");
        assertTrue(destId.isEmpty());
        
        // move
        moveService.processMoveJob(getTransfer("/test3/m1", "/test4/dest1"));

        // source has been moved
        Optional<Long> oldSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(oldSourceId.isEmpty());
        Optional<Long> oldChildId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(oldChildId.isEmpty());
        
        Optional<Long> newSourceId = nodeDao.getNodeId("/test4/dest1");
        assertTrue(newSourceId.isPresent());
        assertEquals(sourceId.get(), newSourceId.get());
        
        Optional<Long> newChildId = nodeDao.getNodeId("/test4/dest1/m2");
        assertTrue(newChildId.isPresent());
        assertEquals(childId.get(), newChildId.get());
        
    }
    
    @Test
    @Order(10)
    public void testMoveToUnexistingParent() {
        User user = mock(User.class);        
        when(user.getName()).thenReturn("user3");
        when(servletRequest.getUserPrincipal()).thenReturn(user);
        
        Optional<Long> sourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(sourceId.isPresent());
        Optional<Long> childId = nodeDao.getNodeId("/test3/m1/m2");
        assertTrue(childId.isPresent());
        
        Optional<Long> destParentId = nodeDao.getNodeId("/test4");
        assertTrue(destParentId.isPresent());
        
        Optional<Long> destCreatemeId = nodeDao.getNodeId("/test4/createme");
        assertTrue(destCreatemeId.isEmpty());
        
        // Rename
        moveService.processMoveJob(getTransfer("/test3/m1", "/test4/createme/dest1"));
        
        Optional<Long> checkSourceId = nodeDao.getNodeId("/test3/m1");
        assertTrue(checkSourceId.isEmpty());
        
        Optional<Long> newCreatemeId = nodeDao.getNodeId("/test4/createme");
        assertTrue(newCreatemeId.isPresent());        
        
        Optional<Long> newSourceId = nodeDao.getNodeId("/test4/createme/dest1");
        assertTrue(newSourceId.isPresent());
        assertEquals(sourceId.get(), newSourceId.get());

        Optional<Long> newChildId = nodeDao.getNodeId("/test4/createme/dest1/m2");
        assertTrue(newChildId.isPresent());
        assertEquals(childId.get(), newChildId.get());        
        
    }     

    private Transfer getTransfer(String vosTarget, String vosDestination) {
        Transfer transfer = new Transfer();
        transfer.setTarget("vos://" + this.authority + vosTarget);
        transfer.setDirection("vos://" + this.authority + vosDestination);
        return transfer;
    }
    
    private ContainerNode getContainerNode(String vosPath, String owner, String writeGroups)
    {
        ContainerNode node = new ContainerNode();
        node.setUri("vos://"+this.authority+vosPath);
        Property ownerProp = new Property();
        ownerProp.setUri(NodeProperties.CREATOR_URI);
        ownerProp.setValue(owner);
        
        Property writeGroupsProp = new Property();
        writeGroupsProp.setUri(NodeProperties.GROUP_WRITE_URI);
        writeGroupsProp.setValue(writeGroups);
        
        node.getProperties().add(ownerProp);
        node.getProperties().add(writeGroupsProp);
        
        return node;
    }

}
