/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace.persistence;

import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.datamodel.NodeUtils;
import it.inaf.oats.vospace.exception.InternalFaultException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import javax.sql.DataSource;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.View;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.junit.jupiter.SpringExtension;

@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = {DataSourceConfig.class})
@TestPropertySource(locations = "classpath:test.properties")
public class NodeDAOTest {

    @Autowired
    private DataSource dataSource;
    private NodeDAO dao;

    @BeforeEach
    public void init() {
        dao = new NodeDAO(dataSource);
    }

    @Test
    public void testCreateNode() {
        DataNode dataNode = new DataNode();

        dataNode.setUri("vos://example.com!vospace/mydata1");
        dataNode.setAccepts(getViews());
        dataNode.setProvides(getViews());
        dataNode.setBusy(true);

        dao.createNode(dataNode);

        DataNode retrievedNode = (DataNode) dao.listNode("/mydata1").get();

        assertEquals(retrievedNode.getAccepts().get(0).getUri(), dataNode.getAccepts().get(0).getUri());
        assertEquals(retrievedNode.getProvides().get(0).getUri(), dataNode.getProvides().get(0).getUri());
    }

    @Test
    public void testListNode() {
        ContainerNode root = (ContainerNode) dao.listNode("/").get();
        assertEquals(4, root.getNodes().size());

        assertEquals("true", NodeProperties.getNodePropertyByURI(root, NodeProperties.PUBLIC_READ_URI));
        assertEquals("0", NodeProperties.getNodePropertyByURI(root, NodeProperties.LENGTH_URI));

        assertEquals("group1 group2", NodeProperties.getNodePropertyByURI(root.getNodes().get(0), NodeProperties.GROUP_READ_URI));

        String bTime = NodeProperties.getNodePropertyByURI(root.getNodes().get(0), NodeProperties.INITIAL_CREATION_TIME_URI);
        assertTrue(bTime.contains("T"));
        assertEquals(bTime, NodeProperties.getNodePropertyByURI(root.getNodes().get(0), NodeProperties.DATE_URI));
    }

    @Test
    public void testGetNodeId() {
        Optional<Long> id1 = dao.getNodeId("/test1");
        assertTrue(id1.isPresent());        
        assertEquals(2, id1.get());
        
        Optional<Long> id2 = dao.getNodeId("/test1/f1");
        assertTrue(id2.isPresent());
        assertEquals(3, id2.get());
        
        Optional<Long> id3 = dao.getNodeId("/pippo123123");
        assertTrue(id3.isEmpty());
    }
    
    @Test
    public void testGetNodeById() {
        Optional<Long> id1 = dao.getNodeId("/test1/f1");
        assertTrue(id1.isPresent());
        
        assertThrows(InternalFaultException.class,
                () -> {
                    dao.getNodeById(id1.get(), true);
                });
        
        Optional<Node> opt1 = dao.getNodeById(id1.get(), false);
        
        assertTrue(opt1.isPresent());
        assertTrue(NodeUtils.getVosPath(opt1.get()).equals("/test1/f1"));
    }    

    @Test
    public void testListNodesInBranch() {
        Optional<Long> id1 = dao.getNodeId("/test1/f1");
        assertTrue(id1.isPresent());
        
        List<Node> result = dao.listNodesInBranch(id1.get(), false);
        assertEquals(3, result.size());
        // Check if list has root node at index 0
        Node root = result.get(0);
        assertEquals("/test1/f1", NodeUtils.getVosPath(root));

        assertThrows(InternalFaultException.class,
                () -> {
                    dao.listNodesInBranch(id1.get(), true);
                });

    }

    @Test
    public void testSetBranchBusy() {
        Optional<Long> rootId = dao.getNodeId("/test1/f1");
        assertTrue(rootId.isPresent());
        
        dao.setBranchBusy(rootId.get(), true);
        List<Node> busyList = dao.listNodesInBranch(rootId.get(), false);
        boolean busyTrue = busyList.stream().allMatch((n) -> {
            if (n instanceof DataNode) {
                return ((DataNode) n).isBusy();
            } else {
                return true;
            }
        }
        );

        assertTrue(busyTrue);

        dao.setBranchBusy(rootId.get(), false);

        busyList = dao.listNodesInBranch(rootId.get(), false);

        boolean busyFalse = busyList.stream().allMatch((n) -> {
            if (n instanceof DataNode) {
                return !((DataNode) n).isBusy();
            } else {
                return true;
            }
        }
        );

        assertTrue(busyFalse);
    }

    @Test
    public void testRenameNode() {
        String oldPath = "/test1/f1";
        String newPath = "/test1/f_pippo";
        String child = "/f2_renamed";
        String oldPathChild = oldPath + child;
        String newPathChild = newPath + child;

        assertTrue(dao.listNode(oldPath).isPresent());
        assertTrue(dao.listNode(oldPathChild).isPresent());
        Optional<Long> rootId = dao.getNodeId(oldPath);
        assertTrue(rootId.isPresent());

        dao.renameNode(rootId.get(), "f_pippo");

        assertTrue(dao.listNode(oldPath).isEmpty());
        assertTrue(dao.listNode(oldPathChild).isEmpty());

        assertTrue(dao.listNode(newPath).isPresent());
        assertTrue(dao.listNode(newPathChild).isPresent());

    }
    
    @Test
    public void testMoveNodeBranch() {
        Optional<Long> optSourceId = dao.getNodeId("/test1/f1");
        assertTrue(optSourceId.isPresent());        
        
        Optional<Long> optSourceId1 = dao.getNodeId("/test1/f1/f2_renamed");
        assertTrue(optSourceId1.isPresent());
        
        Optional<Long> optSourceId2 = dao.getNodeId("/test1/f1/f2_renamed/f3");
        assertTrue(optSourceId2.isPresent());
        
        Optional<Long> optDestId = dao.getNodeId("/test2/f4");
        assertTrue(optDestId.isPresent());
        
        dao.moveNodeBranch(optSourceId.get(), optDestId.get());

        Optional<Long> newOptSourceId = dao.getNodeId("/test1/f1");
        assertTrue(newOptSourceId.isEmpty());
                       
        Optional<Long> dest = dao.getNodeId("/test2/f4/f1");
        assertTrue(dest.isPresent());
        assertEquals(dest.get(), optSourceId.get());
        
        Optional<Long> dest1 = dao.getNodeId("/test2/f4/f1/f2_renamed");        
        assertTrue(dest1.isPresent());
        assertEquals(dest1.get(), optSourceId1.get());
        
        Optional<Long> dest2 = dao.getNodeId("/test2/f4/f1/f2_renamed/f3");
        assertTrue(dest2.isPresent());
        assertEquals(dest2.get(), optSourceId2.get());
        
    }

    @Test
    public void testCountNodeWithPath() {
        assertEquals(1, dao.countNodesWithPath("/"));
        assertEquals(1, dao.countNodesWithPath("/test1"), "Test db has been changed");
        assertEquals(1, dao.countNodesWithPath("/test1/f1"), "Test db has been changed");
        assertEquals(1, dao.countNodesWithPath("/test1/f1/f2_renamed"), "Test db has been changed");
        assertEquals(1, dao.countNodesWithPath("/test1/f1/f2_renamed/f3"), "Test db has been changed");

        assertEquals(1, dao.countNodesWithPath("/test2"), "Test db has been changed");

        assertEquals(1, dao.countNodesWithPath("/test2/f4"), "Test db has been changed");
        assertEquals(1, dao.countNodesWithPath("/test2/f5"), "Test db has been changed");

        assertEquals(0, dao.countNodesWithPath("/pippooo"), "Test db has been changed");
    }

    @Test
    public void testDeleteNode() {

        assertEquals(1, dao.countNodesWithPath("/test1/f1/f2_renamed/f3"), "Test db has been changed");
        dao.deleteNode("/test1");

        assertEquals(0, dao.countNodesWithPath("/test1"));
        assertEquals(0, dao.countNodesWithPath("/test1/f1"));
        assertEquals(0, dao.countNodesWithPath("/test1/f1/f2_renamed"));
        assertEquals(0, dao.countNodesWithPath("/test1/f1/f2_renamed/f3"));

    }

    @Test
    public void testSetNodeLocation() {

        DataNode dataNode = new DataNode();
        dataNode.setUri("vos://example.com!vospace/mydata2");
        dao.createNode(dataNode);

        dao.setNodeLocation("/mydata2", 1, "mydata2");
    }

    @Test
    public void testSetNodeLocationFailure() {
        boolean exception = false;
        try {
            dao.setNodeLocation("/foo", 1, "foo");
        } catch (InternalFaultException ex) {
            exception = true;
        }
        assertTrue(exception);
    }

    @Test
    public void testSetNode() {

        Property publicReadProperty = getProperty(NodeProperties.PUBLIC_READ_URI, String.valueOf(false));

        Node node = new DataNode();
        node.setUri("vos://example.com!vospace/mydata3");
        node.getProperties().add(publicReadProperty);
        dao.createNode(node);

        node = dao.listNode("/mydata3").get();
        assertEquals("false", NodeProperties.getNodePropertyByURI(node, NodeProperties.PUBLIC_READ_URI));

        node.getProperties().clear();
        publicReadProperty.setValue(String.valueOf(true));
        node.getProperties().add(publicReadProperty);

        dao.setNode(node);

        node = dao.listNode("/mydata3").get();
        assertEquals("true", NodeProperties.getNodePropertyByURI(node, NodeProperties.PUBLIC_READ_URI));
    }

    @Test
    public void testSetNodeRecursiveGroup() {

        Property parentGroupRead = getProperty(NodeProperties.GROUP_READ_URI, "group1 group2");
        Property parentGroupWrite = getProperty(NodeProperties.GROUP_WRITE_URI, "group2 group3");

        Node node = new DataNode();
        node.setUri("vos://example.com!vospace/mydata4");
        node.getProperties().add(parentGroupRead);
        node.getProperties().add(parentGroupWrite);
        dao.createNode(node);

        Node child1 = new DataNode();
        child1.setUri("vos://example.com!vospace/mydata4/child1");
        child1.getProperties().add(getProperty(NodeProperties.GROUP_READ_URI, "group3"));
        child1.getProperties().add(getProperty(NodeProperties.GROUP_WRITE_URI, "group3 group4"));
        dao.createNode(child1);

        Node child2 = new DataNode();
        child2.setUri("vos://example.com!vospace/mydata4/child1/child2");
        child2.getProperties().add(getProperty(NodeProperties.GROUP_READ_URI, "group2 group5"));
        child2.getProperties().add(getProperty(NodeProperties.GROUP_WRITE_URI, "group6"));
        dao.createNode(child2);

        parentGroupRead.setValue("group1 group5"); // remove group2; add group5
        parentGroupWrite.setValue("group2 group6"); // remove group3; add group6

        // Recursively set node
        dao.setNode(node, true);

        node = dao.listNode("/mydata4").get();
        child1 = dao.listNode("/mydata4/child1").get();
        child2 = dao.listNode("/mydata4/child1/child2").get();

        checkGroups(NodeProperties.getNodePropertyAsListByURI(node, NodeProperties.GROUP_READ_URI), "group1", "group5");
        checkGroups(NodeProperties.getNodePropertyAsListByURI(node, NodeProperties.GROUP_WRITE_URI), "group2", "group6");

        checkGroups(NodeProperties.getNodePropertyAsListByURI(child1, NodeProperties.GROUP_READ_URI), "group3", "group5");
        checkGroups(NodeProperties.getNodePropertyAsListByURI(child1, NodeProperties.GROUP_WRITE_URI), "group4", "group6");

        checkGroups(NodeProperties.getNodePropertyAsListByURI(child2, NodeProperties.GROUP_READ_URI), "group5");
        checkGroups(NodeProperties.getNodePropertyAsListByURI(child2, NodeProperties.GROUP_WRITE_URI), "group6");
    }

    private Property getProperty(String uri, String value) {
        Property property = new Property();
        property.setUri(uri);
        property.setValue(value);
        return property;
    }

    private void checkGroups(List<String> groups, String... expectedGroups) {
        Set<String> set1 = new HashSet<>(Arrays.asList(expectedGroups));
        Set<String> set2 = new HashSet<>(groups);
        assertEquals(set1, set2);
    }

    private List<View> getViews() {
        View view = new View();
        view.setUri("urn:myview");
        List<View> views = new ArrayList<>();
        views.add(view);
        return views;
    }
}
