package it.inaf.oats.vospace.persistence;

import it.inaf.oats.vospace.JobService;
import java.util.List;
import javax.sql.DataSource;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.uws.v1.ShortJobDescription;
import net.ivoa.xml.vospace.v2.Transfer;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import java.util.Optional;
import java.time.LocalDateTime;
import java.time.Month;
import net.ivoa.xml.uws.v1.Jobs;

@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = {DataSourceConfig.class})
@TestPropertySource(locations = "classpath:test.properties")
public class JobDAOTest {

    @Autowired
    private DataSource dataSource;
    private JobDAO dao;

    @BeforeEach
    public void init() {
        dao = new JobDAO(dataSource);
    }

    @Test
    public void testJob() {

        Transfer transfer = new Transfer();
        transfer.setDirection("pushToVoSpace");
        transfer.setTarget("vos://example.com!vospace/mynode");

        JobSummary job = new JobSummary();
        job.setJobId("123");
        job.setOwnerId("owner");
        job.setPhase(ExecutionPhase.PENDING);
        JobSummary.JobInfo jobInfo = new JobSummary.JobInfo();
        jobInfo.getAny().add(transfer);
        job.setJobInfo(jobInfo);

        dao.createJob(job);

        assertTrue(dao.getJob("123").isPresent());
        assertEquals(ExecutionPhase.PENDING, dao.getJob("123").get().getPhase());

        job.setPhase(ExecutionPhase.EXECUTING);
        dao.updateJob(job);

        assertEquals(ExecutionPhase.EXECUTING, dao.getJob("123").get().getPhase());
    }

    @Test
    public void testJobsList() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobType> directionList = List.of();
        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.ofNullable(null);

        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);

        assertTrue(jobs != null);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertTrue(sjdList != null);
        assertTrue(!sjdList.isEmpty());
        assertTrue(
                sjdList.stream().noneMatch(
                        (i) -> {
                            return i.getPhase().equals(ExecutionPhase.ARCHIVED);
                        }
                )
        );
        assertEquals(3, sjdList.size());
    }

    @Test
    public void testJobsListLimit() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobType> directionList = List.of();
        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.of(2);

        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(2, sjdList.size());

    }

    @Test
    public void testJobsPhase() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList
                = List.of(ExecutionPhase.PENDING, ExecutionPhase.EXECUTING);
        List<JobService.JobType> directionList = List.of();
        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.ofNullable(null);

        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(sjdList.size(), 2);
        assertEquals("pippo5", sjdList.get(0).getId());
        assertEquals("pippo2", sjdList.get(1).getId());
    }

    @Test
    public void testJobsDirection() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobType> directionList
                = List.of(JobService.JobType.pullFromVoSpace,
                        JobService.JobType.pullToVoSpace);

        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.ofNullable(null);
        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(2, sjdList.size());
        assertEquals("pippo3", sjdList.get(0).getId());
        assertEquals("pippo2", sjdList.get(1).getId());
    }

    @Test
    public void testJobsAfter() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobType> directionList = List.of();

        LocalDateTime ldt
                = LocalDateTime.of(2013, Month.FEBRUARY, 7, 18, 15);
        Optional<LocalDateTime> after = Optional.of(ldt);

        Optional<Integer> last = Optional.ofNullable(null);
        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(2, sjdList.size());
        assertEquals("pippo5", sjdList.get(0).getId());
        assertEquals("pippo3", sjdList.get(1).getId());
    }

    @Test
    public void testJobsAllchecks() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of(ExecutionPhase.QUEUED, 
                ExecutionPhase.PENDING);
        List<JobService.JobType> directionList =
                List.of(JobService.JobType.pullFromVoSpace,
                        JobService.JobType.pullToVoSpace);

        LocalDateTime ldt
                = LocalDateTime.of(2013, Month.FEBRUARY, 7, 18, 15);
        Optional<LocalDateTime> after = Optional.of(ldt);

        Optional<Integer> last = Optional.of(2);
        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(1, sjdList.size());
        assertEquals("pippo3", sjdList.get(0).getId());
    }
}
