package it.inaf.oats.vospace;

import it.inaf.oats.vospace.persistence.NodeDAO;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.UnstructuredDataNode;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Mockito.verify;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.put;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.mockito.ArgumentMatchers.any;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;

@SpringBootTest
@AutoConfigureMockMvc
public class CreateNodeControllerTest {

    @MockBean
    private NodeDAO nodeDao;

    @SpyBean
    @Autowired
    private CreateNodeController controller;

    @Autowired
    private MockMvc mockMvc;

    @Test
    public void testFromJsonToXml() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.json");

        mockMvc.perform(put("/nodes/mydata1")
                .content(requestBody)
                .contentType(MediaType.APPLICATION_JSON)
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().isOk());

        verifyArguments();
    }

    @Test
    public void testFromXmlToJson() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.xml");

        mockMvc.perform(put("/nodes/mydata1")
                .content(requestBody)
                .contentType(MediaType.APPLICATION_XML)
                .accept(MediaType.APPLICATION_JSON))
                .andDo(print())
                .andExpect(status().isOk());

        verifyArguments();
    }

    @Test
    public void testFromXmlToXml() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.xml");

        mockMvc.perform(put("/nodes/mydata1")
                .content(requestBody)
                .contentType(MediaType.APPLICATION_XML)
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().isOk());

        verifyArguments();
    }

    @Test
    public void testFromJsonToJson() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.json");

        mockMvc.perform(put("/nodes/mydata1")
                .content(requestBody)
                .contentType(MediaType.APPLICATION_JSON)
                .accept(MediaType.APPLICATION_JSON))
                .andDo(print())
                .andExpect(status().isOk());

        verifyArguments();
    }

    private void verifyArguments() {
        verify(controller).createNode(
                argThat(node -> {
                    UnstructuredDataNode udn = (UnstructuredDataNode) node;
                    Property property = udn.getProperties().get(0);
                    return "vos:UnstructuredDataNode".equals(udn.getType())
                            && "test value".equals(property.getValue())
                            && "ivo://ivoa.net/vospace/core#description".equals(property.getUri());
                }), any());
    }

    protected static String getResourceFileContent(String fileName) throws Exception {
        try ( InputStream in = CreateNodeControllerTest.class.getClassLoader().getResourceAsStream(fileName)) {
            return new String(in.readAllBytes(), StandardCharsets.UTF_8);
        }
    }
}
