package it.inaf.oats.vospace;

import static it.inaf.oats.vospace.CreateNodeControllerTest.getResourceFileContent;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.List;
import java.util.Optional;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.web.servlet.MockMvc;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.when;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.request.MockMvcRequestBuilders;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.put;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;

@SpringBootTest
@ContextConfiguration(classes = {TokenFilterConfig.class})
@TestPropertySource(properties = "spring.main.allow-bean-definition-overriding=true")
@AutoConfigureMockMvc 
public class DeleteNodeControllerTest {    
    
    private static final String URI_PREFIX = "vos://example.com!vospace";
    
    @MockBean
    private NodeDAO nodeDao;

    @Autowired
    private MockMvc mockMvc;

    
    @Test
    public void testDeleteRootNode() throws Exception {
        
        when(nodeDao.listNode(eq("/"))).thenReturn(getRootNode());

        mockMvc.perform(MockMvcRequestBuilders
                        .delete("/nodes")
                        .header("Authorization", "Bearer user2_token"))
                        .andExpect(status().isForbidden());
        
    }
   
    
    @Test
    public void testDeleteFirstLevelNode() throws Exception {
        
        when(nodeDao.listNode(eq("/mynode"))).thenReturn(Optional.of(getWritableDataNode()));
        
        mockMvc.perform(MockMvcRequestBuilders
                        .delete("/nodes/mynode")
                        .header("Authorization", "Bearer user2_token"))
                        .andExpect(status().isOk());
        
    }
       

    
    @Test
    public void testDeleteMoreLevelNode() throws Exception {
        
        when(nodeDao.listNode(eq("/mynode"))).thenReturn(Optional.of(getWritableDataNode("/mynode")));        
        when(nodeDao.listNode(eq("/mynode/middlenode"))).thenReturn(Optional.of(getWritableDataNode("/mynode/middlenode")));        
        when(nodeDao.listNode(eq("/mynode/middlenode/myfile.txt"))).thenReturn(Optional.of(getWritableDataNode("/mynode/middlenode/myfile.txt")));
        
        mockMvc.perform(MockMvcRequestBuilders
                        .delete("/nodes/mynode/middlenode/myfile.txt")
                        .header("Authorization", "Bearer user2_token"))
                        .andExpect(status().isOk());
        
    }

    
    @Test
    public void testLeafNodeNotFound() throws Exception {
        
        when(nodeDao.listNode(eq("/mynode"))).thenReturn(Optional.of(getWritableDataNode("/mynode")));
        
        mockMvc.perform(MockMvcRequestBuilders
                        .delete("/nodes/mynode/notexisting")
                        .header("Authorization", "Bearer user2_token"))
                        .andExpect(status().isNotFound());
        
    }

    
    @Test
    public void testMiddleLevelNodeNotFound() throws Exception {
        
        when(nodeDao.listNode(eq("/mynode"))).thenReturn(Optional.of(getWritableDataNode("/mynode")));     
        when(nodeDao.listNode(eq("/mynode/middlenode/myfile.txt"))).thenReturn(Optional.of(getWritableDataNode("/mynode/middlenode/myfile.txt")));
        
        mockMvc.perform(MockMvcRequestBuilders
                        .delete("/mynode/middlenode/myfile.txt")
                        .header("Authorization", "Bearer user2_token"))
                        .andExpect(status().isNotFound());
        
    }

    
    @Test
    public void testLinkNodeLeafDelete() throws Exception {
        
        when(nodeDao.listNode(eq("/mynode"))).thenReturn(Optional.of(getWritableLinkNode("/mynode")));
        
        mockMvc.perform(MockMvcRequestBuilders
                        .delete("/nodes/mynode")
                        .header("Authorization", "Bearer user2_token"))
                        .andExpect(status().isOk());
        
    }

    
    @Test
    public void testMiddleLevelLinkNodeDelete() throws Exception {
        
        when(nodeDao.listNode(eq("/mynode"))).thenReturn(Optional.of(getWritableDataNode("/mynode")));        
        when(nodeDao.listNode(eq("/mynode/middlenode"))).thenReturn(Optional.of(getWritableLinkNode("/mynode/middlenode")));        
        when(nodeDao.listNode(eq("/mynode/middlenode/myfile.txt"))).thenReturn(Optional.of(getWritableDataNode("/mynode/middlenode/myfile.txt")));
        
        mockMvc.perform(MockMvcRequestBuilders
                        .delete("/nodes/mynode/middlenode/myfile.txt")
                        .header("Authorization", "Bearer user2_token"))
                        .andExpect(status().isBadRequest());
        
    }
    

    @Test
    public void testDeleteMoreLevelNodeNotAllowed() throws Exception {
        
        when(nodeDao.listNode(eq("/mynode"))).thenReturn(Optional.of(getWritableDataNode("/mynode")));        
        when(nodeDao.listNode(eq("/mynode/middlenode"))).thenReturn(Optional.of(getWritableDataNode("/mynode/middlenode")));        
        when(nodeDao.listNode(eq("/mynode/middlenode/myfile.txt"))).thenReturn(Optional.of(getWritableDataNode("/mynode/middlenode/myfile.txt")));
        
        mockMvc.perform(MockMvcRequestBuilders
                        .delete("/nodes/mynode/middlenode/myfile.txt"))
                        .andExpect(status().isForbidden());
        
    }
    
    
    private Optional<Node> getRootNode() {
        ContainerNode root = new ContainerNode();
        root.setUri(URI_PREFIX + "/");
        root.getNodes().add(getWritableDataNode());
        return Optional.of(root);
    }
    
    

    private Node getWritableDataNode() {
        DataNode node = new DataNode();
        List nodeProperties = node.getProperties();
        Property groupWriteProp = new Property();
        groupWriteProp.setUri(NodeProperties.GROUP_WRITE_URI);
        groupWriteProp.setValue("group1");
        nodeProperties.add(groupWriteProp);
        node.setUri(URI_PREFIX + "/mynode");
        return node;
    }
    
    
    private Node getWritableDataNode(String path) {
        DataNode node = new DataNode();
        List nodeProperties = node.getProperties();
        Property groupWriteProp = new Property();
        groupWriteProp.setUri(NodeProperties.GROUP_WRITE_URI);
        groupWriteProp.setValue("group1");
        nodeProperties.add(groupWriteProp);
        node.setUri(URI_PREFIX + path);
        return node;
    }
    

    private LinkNode getWritableLinkNode(String path) {
        LinkNode myNode = new LinkNode();
        // Set parent node address at /
        myNode.setUri("vos://example.com!vospace" + path);
        // Set groupwrite property
        Property groups = new Property();
        groups.setUri("ivo://ivoa.net/vospace/core#groupwrite");
        groups.setValue("group1");
        myNode.setProperties(List.of(groups));
        return myNode;
    }

}
