package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import net.ivoa.xml.vospace.v2.Node;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.beans.factory.annotation.Autowired;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.List;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.PutMapping;
import it.inaf.oats.vospace.exception.*;
import java.util.stream.Collectors;
import java.util.Arrays;

@RestController
public class CreateNodeController extends BaseNodeController {

    @Autowired
    private NodeDAO nodeDao;

    @Value("${vospace-authority}")
    private String authority;

    @PutMapping(value = {"/nodes", "/nodes/**"},
            consumes = {MediaType.APPLICATION_XML_VALUE, MediaType.APPLICATION_JSON_VALUE},
            produces = {MediaType.APPLICATION_XML_VALUE, MediaType.APPLICATION_JSON_VALUE})
    public Node createNode(@RequestBody Node node, User principal) {

        String path = getPath();

        List<String> userGroups = principal.getGroups();

        if (!isValidURI(node.getUri()))
            throw new InvalidURIException(node.getUri());
        
        if(!isUrlConsistentWithPayloadURI(node.getUri(), path)) {
            throw new InvalidURIException(node.getUri(), path);
        }

        // This checks if the user is trying to insert the root node at "/" too
        if (nodeDao.listNode(path).isPresent()) {
            throw new DuplicateNodeException(path);
        }

        // Retrieve parent node 
        Node parentNode = nodeDao.listNode(getParentPath(path))
                .orElseThrow(() -> new ContainerNotFoundException(getParentPath(path)));

        List<String> groupWritePropValues = parentNode.getProperties().stream()
                .filter((i) -> i.getUri()
                .equals("ivo://ivoa.net/vospace/core#groupwrite"))
                .map((i) -> i.getValue())
                .collect(Collectors.toList());

        if (groupWritePropValues.isEmpty()) {
            throw new PermissionDeniedException(path);
        }

        List<String> nodeGroups
                = Arrays.asList(groupWritePropValues.get(0).split(",",-1));

        if (!nodeGroups.stream().anyMatch((i) -> userGroups.contains(i))) {
            throw new PermissionDeniedException(path);
        }

        // Check if parent node is a LinkNode and if so throw exception
        if (parentNode.getType().equals("vos:LinkNode")) {
            throw new LinkFoundException(getParentPath(path));
        }

        nodeDao.createNode(node);
        return node;
    }
    
    // Assuming that this service implementation uses only ! as a separator
    // in the authority part of the URI
    private boolean isValidURI(String nodeURI)
    {
        String parsedAuthority;
        if(!nodeURI.startsWith("vos://"))
        {
            return false;
        } else {
            parsedAuthority = nodeURI.replaceAll("vos://", "").split("/",-1)[0];
        }
        
        if(parsedAuthority.isEmpty() || 
                !parsedAuthority.replace("~","!").equals(authority))
            return false;
        
        return true;        
    }

    private boolean isUrlConsistentWithPayloadURI(String nodeURI, String path) {
        // It's assumed that nodeURI has been validated to be in the expected 
        // form vos://authority[!~]somepath/mynode..."

        return nodeURI.replaceAll("vos://[^/]+", "").equals(path);

    }

    private String getParentPath(String path) {
        String[] parsedPath = path.split("/");

        StringBuilder sb = new StringBuilder();

        for (int i = 0; i < parsedPath.length - 1; i++) {
            sb.append("/").append(parsedPath[i]);
        }

        return sb.toString();
    }
}
