package it.inaf.oats.vospace;

import it.inaf.oats.vospace.persistence.NodeDAO;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import net.ivoa.xml.vospace.v2.Property;
import net.ivoa.xml.vospace.v2.UnstructuredDataNode;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.springframework.http.MediaType;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.put;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import static org.mockito.ArgumentMatchers.any;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.LinkNode;
import java.util.List;
import it.inaf.ia2.aa.data.User;
import java.util.Optional;

@SpringBootTest
@AutoConfigureMockMvc(addFilters = false)
public class CreateNodeControllerTest {

    @MockBean
    private NodeDAO nodeDao;

    @SpyBean
    @Autowired
    private CreateNodeController controller;

    @Autowired
    private MockMvc mockMvc;

    private ContainerNode getContainerParentNode(String path) {
        ContainerNode parentNode = new ContainerNode();
        // Set parent node address at /
        parentNode.setUri("vos://example.com!vospace" + path);
        // Set groupwrite property
        Property groups = new Property();
        groups.setUri("ivo://ivoa.net/vospace/core#groupwrite");
        groups.setValue("test1,test2");
        parentNode.setProperties(List.of(groups));
        return parentNode;
    }

    private LinkNode getLinkParentNode(String path) {
        LinkNode parentNode = new LinkNode();
        // Set parent node address at /
        parentNode.setUri("vos://example.com!vospace" + path);
        // Set groupwrite property
        Property groups = new Property();
        groups.setUri("ivo://ivoa.net/vospace/core#groupwrite");
        groups.setValue("test1,test2");
        parentNode.setProperties(List.of(groups));
        return parentNode;
    }

    private User getUser() {
        User user = new User();
        user.setGroups(List.of("test1"));
        return user;
    }

    @Test
    public void testFromJsonToXml() throws Exception {
        String requestBody
                = getResourceFileContent("create-unstructured-data-node.json");

        when(nodeDao.listNode(eq("/")))
                .thenReturn(Optional.of(getContainerParentNode("/")));

        mockMvc.perform(put("/nodes/mydata1")
                .principal(getUser())
                .content(requestBody)
                .contentType(MediaType.APPLICATION_JSON)
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().isOk());

        verifyArguments();
    }

    @Test
    public void testFromXmlToJson() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.xml");

        when(nodeDao.listNode(eq("/")))
                .thenReturn(Optional.of(getContainerParentNode("/")));

        mockMvc.perform(put("/nodes/mydata1")
                .principal(getUser())
                .content(requestBody)
                .contentType(MediaType.APPLICATION_XML)
                .accept(MediaType.APPLICATION_JSON))
                .andDo(print())
                .andExpect(status().isOk());

        verifyArguments();
    }

    @Test
    public void testFromXmlToXml() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.xml");

        when(nodeDao.listNode(eq("/")))
                .thenReturn(Optional.of(getContainerParentNode("/")));

        mockMvc.perform(put("/nodes/mydata1")
                .principal(getUser())
                .content(requestBody)
                .contentType(MediaType.APPLICATION_XML)
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().isOk());

        verifyArguments();
    }

    @Test
    public void testFromJsonToJson() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.json");

        when(nodeDao.listNode(eq("/")))
                .thenReturn(Optional.of(getContainerParentNode("/")));

        mockMvc.perform(put("/nodes/mydata1")
                .principal(getUser())
                .content(requestBody)
                .contentType(MediaType.APPLICATION_JSON)
                .accept(MediaType.APPLICATION_JSON))
                .andDo(print())
                .andExpect(status().isOk());

        verifyArguments();
    }

    @Test
    public void testURIConsistence() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.xml");

        when(nodeDao.listNode(eq("/")))
                .thenReturn(Optional.of(getContainerParentNode("/")));

        User user = new User();
        user.setGroups(List.of("test3", "test4"));
        
        mockMvc.perform(put("/nodes/mydata2")
                .principal(user)
                .content(requestBody)
                .contentType(MediaType.APPLICATION_XML)
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().is4xxClientError());

        verifyArguments();

    }

    @Test
    public void testNodeAlreadyExisting() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.xml");

        when(nodeDao.listNode(eq("/")))
                .thenReturn(Optional.of(getContainerParentNode("/")));
        when(nodeDao.listNode(eq("/mydata1")))
                .thenReturn(Optional.of(getContainerParentNode("/mydata1")));

        mockMvc.perform(put("/nodes/mydata1")
                .principal(getUser())
                .content(requestBody)
                .contentType(MediaType.APPLICATION_XML)
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().is4xxClientError());

        verifyArguments();
    }

    @Test
    public void testContainerNotFound() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.xml");

        when(nodeDao.listNode(eq("/")))
                .thenReturn(Optional.ofNullable(null));

        mockMvc.perform(put("/nodes/mydata1")
                .principal(getUser())
                .content(requestBody)
                .contentType(MediaType.APPLICATION_XML)
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().is4xxClientError());

        verifyArguments();
    }
    
    @Test
    public void testLinkNodeFound() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.xml");

        when(nodeDao.listNode(eq("/")))
                .thenReturn(Optional.of(getLinkParentNode("/")));

        mockMvc.perform(put("/nodes/mydata1")
                .principal(getUser())
                .content(requestBody)
                .contentType(MediaType.APPLICATION_XML)
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().is4xxClientError());

        verifyArguments();
    }
    
    @Test
    public void testPermissionDenied() throws Exception {
        String requestBody = getResourceFileContent("create-unstructured-data-node.xml");

        when(nodeDao.listNode(eq("/")))
                .thenReturn(Optional.of(getLinkParentNode("/")));

        mockMvc.perform(put("/nodes/mydata1")
                .principal(getUser())
                .content(requestBody)
                .contentType(MediaType.APPLICATION_XML)
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().is4xxClientError());

        verifyArguments();
    }

    private void verifyArguments() {
        verify(controller).createNode(
                argThat(node -> {
                    UnstructuredDataNode udn = (UnstructuredDataNode) node;
                    Property property = udn.getProperties().get(0);
                    return "vos:UnstructuredDataNode".equals(udn.getType())
                            && "test value".equals(property.getValue())
                            && "ivo://ivoa.net/vospace/core#description".equals(property.getUri());
                }), any());
    }

    protected static String getResourceFileContent(String fileName) throws Exception {
        try (InputStream in = CreateNodeControllerTest.class.getClassLoader().getResourceAsStream(fileName)) {
            return new String(in.readAllBytes(), StandardCharsets.UTF_8);
        }
    }
}
