/*
 * This file is part of vospace-rest
 * Copyright (C) 2021 Istituto Nazionale di Astrofisica
 * SPDX-License-Identifier: GPL-3.0-or-later
 */
package it.inaf.oats.vospace;

import it.inaf.oats.vospace.persistence.CollectionsDAO;
import it.inaf.oats.vospace.datamodel.collections.NodeCollection;
import it.inaf.oats.vospace.exception.PermissionDeniedException;
import java.util.ArrayList;
import java.util.List;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.EnableTransactionManagement;

/**
 *
 * @author Nicola Fulvio Calabria <nicola.calabria at inaf.it>
 */
@Service
@EnableTransactionManagement
public class CollectionsService {

    @Autowired
    private CollectionsDAO collectionsDAO;

    public List<NodeCollection> listCollections(String userId) {

        List<NodeCollection> result = new ArrayList<>();

        // If user is not authenticated simply return an empty list
        if (isUserAuthenticated(userId)) {
            result.addAll(
                    collectionsDAO.getUserNodeCollections(userId));
        } else {
            throw new PermissionDeniedException("Authentication required");
        }

        return result;

    }

    public void createNewCollection(String collectionTitle, String userId) {
        if (isUserAuthenticated(userId)) {
            collectionsDAO.createNewCollection(collectionTitle, userId);
        } else {
            throw new PermissionDeniedException("Authentication required");
        }
    }

    public void deleteCollectionById(Long collectionId, String userId) {
        if(isUserAuthenticated(userId))
        {
            collectionsDAO.deleteCollection(collectionId, userId);
            // TODO: throw exceptions if collection not found or deletion requested
            // by someone who is not the owner
        } else {
            throw new PermissionDeniedException("Authentication required");
        }
    }
   
    private boolean isUserAuthenticated(String userId) {
        return userId != null
                && !userId.equals("anonymous");
    }

}
