package it.inaf.oats.vospace;

import it.inaf.ia2.aa.data.User;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import net.ivoa.xml.vospace.v2.Node;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.beans.factory.annotation.Autowired;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.List;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.PutMapping;
import it.inaf.oats.vospace.exception.*;
import java.util.stream.Collectors;

@RestController
public class CreateNodeController extends BaseNodeController {

    @Autowired
    private NodeDAO nodeDao;

    @Value("${vospace-authority}")
    private String authority;

    @PutMapping(value = {"/nodes", "/nodes/**"},
            consumes = {MediaType.APPLICATION_XML_VALUE, MediaType.TEXT_XML_VALUE, MediaType.APPLICATION_JSON_VALUE},
            produces = {MediaType.APPLICATION_XML_VALUE, MediaType.TEXT_XML_VALUE, MediaType.APPLICATION_JSON_VALUE})
    public Node createNode(@RequestBody Node node, User principal) {

        String path = getPath();

        // Validate payload node URI
        if (!isValidURI(node.getUri())) {
            throw new InvalidURIException(node.getUri());
        }

        // Check if payload URI is consistent with http request
        if (!isUrlConsistentWithPayloadURI(node.getUri(), path)) {
            throw new InvalidURIException(node.getUri(), path);
        }

        // Check if another node is already present at specified path
        // This checks if the user is trying to insert the root node at "/" too
        if (nodeDao.listNode(path).isPresent()) {
            throw new DuplicateNodeException(path);
        }

        // Retrieve parent node 
        Node parentNode = nodeDao.listNode(getParentPath(path))
                .orElseThrow(() -> new ContainerNotFoundException(getParentPath(path)));

        // Check if parent node is not a Container node and in case throw
        // appropriate exception
        if (!parentNode.getType().equals("vos:ContainerNode")) {
            if (parentNode.getType().equals("vos:LinkNode")) {
                throw new LinkFoundException(getParentPath(path));
            } else {
                throw new ContainerNotFoundException(getParentPath(path));
            }
        }

        // First check if parent node creator is == userid
        List<String> nodeOwner
                = NodeProperties.getNodePropertyByURI(
                        parentNode, NodeProperties.CREATOR_URI);

        if (nodeOwner == null
                || nodeOwner.isEmpty()
                || !nodeOwner.get(0).equals(principal.getName())) {
            // Node owner check has failed: let's check if user can write
            // due to group privileges

            List<String> userGroups = principal.getGroups();

            // If the user doesn't belong to any groups throw exception
            if (userGroups == null || userGroups.isEmpty()) {
                throw new PermissionDeniedException(path);
            }

            List<String> groupWritePropValues
                    = NodeProperties.getNodePropertyByURI(parentNode,
                            "ivo://ivoa.net/vospace/core#groupwrite");

            // If groupwrite property is absent in Parent Node throw exception
            if (groupWritePropValues == null
                    || groupWritePropValues.isEmpty()) {
                throw new PermissionDeniedException(path);
            }

            List<String> nodeGroups
                    = NodeProperties.parsePropertyStringToList(groupWritePropValues.get(0));

            if (nodeGroups.isEmpty()
                    || !nodeGroups.stream()
                            .anyMatch((i) -> userGroups.contains(i))) {
                throw new PermissionDeniedException(path);
            }

        }

        nodeDao.createNode(node);

        return node;
    }

    // Assuming that this service implementation uses only ! as a separator
    // in the authority part of the URI
    private boolean isValidURI(String nodeURI) {
        String parsedAuthority;
        if (!nodeURI.startsWith("vos://")) {
            return false;
        } else {
            parsedAuthority = nodeURI.replaceAll("vos://", "").split("/", -1)[0];
        }

        if (parsedAuthority.isEmpty()
                || !parsedAuthority.replace("~", "!").equals(authority)) {
            return false;
        }

        return true;
    }

    private boolean isUrlConsistentWithPayloadURI(String nodeURI, String path) {
        // It's assumed that nodeURI has been validated to be in the expected 
        // form vos://authority[!~]somepath/mynode..."

        return nodeURI.replaceAll("vos://[^/]+", "").equals(path);

    }

    

}
