package it.inaf.oats.vospace.persistence;

import it.inaf.oats.vospace.JobService;
import java.util.List;
import javax.sql.DataSource;
import net.ivoa.xml.uws.v1.ExecutionPhase;
import net.ivoa.xml.uws.v1.JobSummary;
import net.ivoa.xml.uws.v1.ShortJobDescription;
import net.ivoa.xml.vospace.v2.Transfer;
import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertTrue;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.context.junit.jupiter.SpringExtension;
import java.util.Optional;
import java.time.LocalDateTime;
import java.time.Month;
import net.ivoa.xml.uws.v1.ErrorSummary;
import net.ivoa.xml.uws.v1.Jobs;
import it.inaf.oats.vospace.exception.ErrorSummaryFactory;
import it.inaf.oats.vospace.exception.PermissionDeniedException;

@ExtendWith(SpringExtension.class)
@ContextConfiguration(classes = {DataSourceConfig.class})
@TestPropertySource(locations = "classpath:test.properties")
public class JobDAOTest {

    @Autowired
    private DataSource dataSource;
    private JobDAO dao;

    @BeforeEach
    public void init() {
        dao = new JobDAO(dataSource);
    }

    private JobSummary getJob() {
        Transfer transfer = new Transfer();
        transfer.setDirection("pushToVoSpace");
        transfer.setTarget("vos://example.com!vospace/mynode");

        JobSummary job = new JobSummary();
        job.setJobId("123");
        job.setOwnerId("owner");
        job.setPhase(ExecutionPhase.PENDING);
        JobSummary.JobInfo jobInfo = new JobSummary.JobInfo();
        jobInfo.getAny().add(transfer);
        job.setJobInfo(jobInfo);

        return job;
    }

    // don't want to override equals method just for testing without a use case
    private boolean areEqual(ErrorSummary a, ErrorSummary b) {     
        return (a.getMessage().equals(b.getMessage())
                && a.getType().equals(b.getType())
                && a.isHasDetail() == b.isHasDetail()
                && a.getDetailMessage().equals(b.getDetailMessage()));
    }

    @Test
    public void testJob() {

        JobSummary job = getJob();

        dao.createJob(job);

        assertTrue(dao.getJob("123").isPresent());
        assertEquals(ExecutionPhase.PENDING, dao.getJob("123").get().getPhase());

        // uses the job retrieved from DAO to perform the update (reproduced a bug in job update)
        job = dao.getJob("123").get();

        job.setPhase(ExecutionPhase.EXECUTING);
        dao.updateJob(job);

        assertEquals(ExecutionPhase.EXECUTING, dao.getJob("123").get().getPhase());
    }

    @Test
    public void testCreateJobWithError() {
        JobSummary job = getJob();

        job.setPhase(ExecutionPhase.ERROR);

        // Generate it from exception        
        ErrorSummary errorSummary
                = ErrorSummaryFactory.newErrorSummary(
                        new PermissionDeniedException("/pippo1/pippo2"));
        
        // Check if properly generated
        assertTrue(errorSummary.isHasDetail());
        assertEquals("PermissionDenied. Path: /pippo1/pippo2", errorSummary.getDetailMessage());

        job.setErrorSummary(errorSummary);

        dao.createJob(job);

        // Retrieve it back        
        Optional<JobSummary> retrievedJobOpt = dao.getJob(job.getJobId());
        assertTrue(retrievedJobOpt.isPresent());

        JobSummary retrievedJob = retrievedJobOpt.get();
        assertEquals(ExecutionPhase.ERROR, retrievedJob.getPhase());
        assertTrue(areEqual(job.getErrorSummary(), retrievedJob.getErrorSummary()));

    }

    @Test
    public void testUpdateJobWithError() {
        JobSummary job = getJob();

        dao.createJob(job);

        job.setPhase(ExecutionPhase.ERROR);
        // Generate it from exception        
        ErrorSummary errorSummary
                = ErrorSummaryFactory.newErrorSummary(
                        new PermissionDeniedException("/pippo1/pippo2"));
        
        // Check if properly generated        
        assertTrue(errorSummary.isHasDetail());
        assertEquals("PermissionDenied. Path: /pippo1/pippo2", errorSummary.getDetailMessage());
                
        job.setErrorSummary(errorSummary);

        dao.updateJob(job);

        // Retrieve it back        
        Optional<JobSummary> retrievedJobOpt = dao.getJob(job.getJobId());
        assertTrue(retrievedJobOpt.isPresent());

        JobSummary retrievedJob = retrievedJobOpt.get();
        assertEquals(ExecutionPhase.ERROR, retrievedJob.getPhase());
        assertTrue(areEqual(job.getErrorSummary(), retrievedJob.getErrorSummary()));
    }

    @Test
    public void testJobsList() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobDirection> directionList = List.of();
        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.ofNullable(null);

        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);

        assertTrue(jobs != null);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertTrue(sjdList != null);
        assertTrue(!sjdList.isEmpty());
        assertTrue(
                sjdList.stream().noneMatch(
                        (i) -> {
                            return i.getPhase().equals(ExecutionPhase.ARCHIVED);
                        }
                )
        );
        assertEquals(3, sjdList.size());
    }

    @Test
    public void testJobsListLimit() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobDirection> directionList = List.of();
        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.of(2);

        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(2, sjdList.size());

    }

    @Test
    public void testJobsPhase() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList
                = List.of(ExecutionPhase.PENDING, ExecutionPhase.EXECUTING);
        List<JobService.JobDirection> directionList = List.of();
        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.ofNullable(null);

        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(sjdList.size(), 2);
        assertEquals("pippo5", sjdList.get(0).getId());
        assertEquals("pippo2", sjdList.get(1).getId());
    }

    @Test
    public void testJobsDirection() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobDirection> directionList
                = List.of(JobService.JobDirection.pullFromVoSpace,
                        JobService.JobDirection.pullToVoSpace);

        Optional<LocalDateTime> after = Optional.ofNullable(null);
        Optional<Integer> last = Optional.ofNullable(null);
        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(2, sjdList.size());
        assertEquals("pippo3", sjdList.get(0).getId());
        assertEquals("pippo2", sjdList.get(1).getId());
    }

    @Test
    public void testJobsAfter() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of();
        List<JobService.JobDirection> directionList = List.of();

        LocalDateTime ldt
                = LocalDateTime.of(2013, Month.FEBRUARY, 7, 18, 15);
        Optional<LocalDateTime> after = Optional.of(ldt);

        Optional<Integer> last = Optional.ofNullable(null);
        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(2, sjdList.size());
        assertEquals("pippo5", sjdList.get(0).getId());
        assertEquals("pippo3", sjdList.get(1).getId());
    }

    @Test
    public void testJobsAllchecks() {
        // Check no arguments
        String user = "user1";
        List<ExecutionPhase> phaseList = List.of(ExecutionPhase.QUEUED,
                ExecutionPhase.PENDING);
        List<JobService.JobDirection> directionList
                = List.of(JobService.JobDirection.pullFromVoSpace,
                        JobService.JobDirection.pullToVoSpace);

        LocalDateTime ldt
                = LocalDateTime.of(2013, Month.FEBRUARY, 7, 18, 15);
        Optional<LocalDateTime> after = Optional.of(ldt);

        Optional<Integer> last = Optional.of(2);
        Jobs jobs = dao.getJobs(user, phaseList, directionList, after, last);
        List<ShortJobDescription> sjdList = jobs.getJobref();
        assertEquals(1, sjdList.size());
        assertEquals("pippo3", sjdList.get(0).getId());
    }
}
