package it.inaf.oats.vospace;

import com.fasterxml.jackson.databind.ObjectMapper;
import static it.inaf.oats.vospace.VOSpaceXmlTestUtil.loadDocument;
import it.inaf.oats.vospace.datamodel.NodeProperties;
import it.inaf.oats.vospace.persistence.NodeDAO;
import java.util.Optional;
import net.ivoa.xml.vospace.v2.ContainerNode;
import net.ivoa.xml.vospace.v2.DataNode;
import net.ivoa.xml.vospace.v2.Node;
import net.ivoa.xml.vospace.v2.Property;
import static org.junit.jupiter.api.Assertions.assertEquals;
import org.junit.jupiter.api.Test;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.web.servlet.AutoConfigureMockMvc;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.http.MediaType;
import org.springframework.test.context.ContextConfiguration;
import org.springframework.test.context.TestPropertySource;
import org.springframework.test.web.servlet.MockMvc;
import static org.springframework.test.web.servlet.request.MockMvcRequestBuilders.get;
import static org.springframework.test.web.servlet.result.MockMvcResultHandlers.print;
import static org.springframework.test.web.servlet.result.MockMvcResultMatchers.status;
import org.w3c.dom.Document;

@SpringBootTest
@AutoConfigureMockMvc
@ContextConfiguration(classes = {TokenFilterConfig.class})
@TestPropertySource(properties = "spring.main.allow-bean-definition-overriding=true")
public class ListNodeControllerTest {

    private static final String URI_PREFIX = "vos://example.com!vospace";

    private static final ObjectMapper MAPPER = new ObjectMapper();
    
    @MockBean
    private NodeDAO dao;

    @Autowired
    private MockMvc mockMvc;

    @Test
    public void testRootXml() throws Exception {

        when(dao.listNode(eq("/"))).thenReturn(getRootNode());

        String xml = mockMvc.perform(get("/nodes")
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().isOk())
                .andReturn().getResponse().getContentAsString();

        Document doc = loadDocument(xml);
        assertEquals("vos:node", doc.getDocumentElement().getNodeName());
        assertEquals("vos:ContainerNode", doc.getDocumentElement().getAttribute("xsi:type"));

        verify(dao, times(1)).listNode(eq("/"));
    }

    @Test
    public void testNodeXml() throws Exception {

        when(dao.listNode(eq("/mynode"))).thenReturn(Optional.of(getDataNode()));

        String xml = mockMvc.perform(get("/nodes/mynode")
                .accept(MediaType.APPLICATION_XML))
                .andDo(print())
                .andExpect(status().isOk())
                .andReturn().getResponse().getContentAsString();

        Document doc = loadDocument(xml);
        assertEquals("vos:node", doc.getDocumentElement().getNodeName());
        assertEquals("vos:DataNode", doc.getDocumentElement().getAttribute("xsi:type"));

        verify(dao, times(1)).listNode(eq("/mynode"));
    }

    @Test
    public void testNodeNotFound() throws Exception {
        mockMvc.perform(get("/nodes/mynode")
                .accept(MediaType.APPLICATION_XML))
                .andExpect(status().isNotFound());
    }
    
    @Test
    public void testPermissionDeniedUser() throws Exception {
        Node node = getDataNodeByOwnership("user2","group1");
        
        when(dao.listNode(eq("/mynode"))).thenReturn(Optional.of(node));
        
        mockMvc.perform(get("/nodes/mynode")
                .header("Authorization", "Bearer user1_token")
                .accept(MediaType.APPLICATION_XML))
                .andExpect(status().is4xxClientError());       
    }
    
    @Test
    public void testGrantedByGroup() throws Exception {
        Node node = getDataNodeByOwnership("user1","group1");
        
        when(dao.listNode(eq("/mynode"))).thenReturn(Optional.of(node));
        
        mockMvc.perform(get("/nodes/mynode")
                .header("Authorization", "Bearer user2_token")
                .accept(MediaType.APPLICATION_XML))
                .andExpect(status().is2xxSuccessful());       
    }
    
    //@Test
    public void testListOnlyReadableChildren() throws Exception {

        ContainerNode node = new ContainerNode();
        node.setUri(URI_PREFIX + "/mynode");
        Property creator = new Property();
        creator.setUri(NodeProperties.CREATOR_URI);
        creator.setValue("user1");
        node.getProperties().add(creator);

        // hidden node
        Node child1 = getDataNodeByOwnership("user2", "group5");
        child1.setUri(URI_PREFIX + "/mynode/child1");
        node.getNodes().add(child1);
        
        // visible node
        Node child2 = getDataNodeByOwnership("user1", "group1");
        child2.setUri(URI_PREFIX + "/mynode/child2");
        node.getNodes().add(child2);
        
        assertEquals(2, node.getNodes().size());

        when(dao.listNode(eq("/mynode"))).thenReturn(Optional.of(node));

        String response = mockMvc.perform(get("/nodes/mynode")
                .header("Authorization", "Bearer user1_token")
                .accept(MediaType.APPLICATION_JSON))
                .andExpect(status().is2xxSuccessful())
                .andReturn().getResponse().getContentAsString();
        
        ContainerNode result = MAPPER.readValue(response, ContainerNode.class);
        
        assertEquals(1, result.getNodes().size());
        assertEquals(URI_PREFIX + "/mynode/child2", result.getNodes().get(0).getUri());
    }

    private Optional<Node> getRootNode() {
        ContainerNode root = new ContainerNode();
        root.setUri(URI_PREFIX + "/");
        Property publicProperty = new Property();
        publicProperty.setUri(NodeProperties.PUBLIC_READ_URI);
        publicProperty.setValue("true");
        root.getProperties().add(publicProperty);       
        
        root.getNodes().add(getDataNode());
        return Optional.of(root);
    }

    private Node getDataNode() {
        DataNode node = new DataNode();
        node.setUri(URI_PREFIX + "/mynode");
        Property publicProperty = new Property();
        publicProperty.setUri(NodeProperties.PUBLIC_READ_URI);
        publicProperty.setValue("true");
        node.getProperties().add(publicProperty);

        return node;
    }
    
    private Node getDataNodeByOwnership(String ownerID, String group)
    {
        DataNode node = new DataNode();
        node.setUri(URI_PREFIX + "/mynode");
        // Set owner
        Property creatorProperty = new Property();
        creatorProperty.setUri(NodeProperties.CREATOR_URI);
        creatorProperty.setValue(ownerID);
        node.getProperties().add(creatorProperty);
        // set group  
        Property readGroupProperty = new Property();
        readGroupProperty.setUri(NodeProperties.GROUP_READ_URI);
        readGroupProperty.setValue(group);
        node.getProperties().add(readGroupProperty);
        
        return node;       
    }
}
