import pika
import uuid
import json
import sys


class AMQPClient(object):

    def __init__(self):
        self.rpcQueue = "store_job_queue"
        self.connection = pika.BlockingConnection(pika.ConnectionParameters(host = "rabbitmq"))
        self.channel = self.connection.channel()
        result = self.channel.queue_declare(queue = '', exclusive = True)
        self.callback_queue = result.method.queue
        self.channel.basic_consume(queue = self.callback_queue, on_message_callback = self.on_response, auto_ack = True)

    def on_response(self, ch, method, props, body):
        if self.corr_id == props.correlation_id:
            self.response = json.loads(body)

    def call(self, msg):
        self.response = None
        self.corr_id = str(uuid.uuid4())
        self.channel.basic_publish(exchange = '',
                                   routing_key = self.rpcQueue,
                                   properties = pika.BasicProperties(reply_to = self.callback_queue,
                                                                     correlation_id = self.corr_id,
                                                                    ),
                                   body = json.dumps(msg))

        while self.response is None:
            self.connection.process_data_events()
        return self.response


    def help(self):
        sys.exit("""
NAME
       dataArchiverCli.py

SYNOPSYS
       python3.x dataArchiverCli.py COMMAND USERNAME

DESCRIPTION
       The purpose of this client application is to notify to the VOSpace backend that
       data is ready to be saved somewhere.
       
       The client accepts only one command at a time. This command is mandatory.
       A list of supported commands is shown here below:

       cstore
              performs a 'cold storage' request, data will be saved on the tape library

       hstore
              performs a 'hot storage' request, data will be saved on a standard server

       The client also needs to know the username associated to a storage request process.
       The username must be the same used for accessing the transfer node.
    """)

    def store(self, cmd, username):
        request_type = cmd.upper()
        storeRequest = { "requestType": request_type, "userName": username }
        print(f"\nSending {request_type} request...")
        storeResponse = self.call(storeRequest)
        if "responseType" not in storeResponse:
            sys.exit("FATAL: Malformed response, store acknowledge expected.\n")
        elif storeResponse["responseType"] == "STORE_ACK":
            print("\nWARNING!!! WARNING!!! WARNING!!! WARNING!!! WARNING!!!")
            print("If you confirm, all your data on the transfer node will be")
            print("available in read-only mode for all the time the archiving")
            print("process is running.")
            print("WARNING!!! WARNING!!! WARNING!!! WARNING!!! WARNING!!!\n")
            confirm = None
            while not confirm in [ "yes", "no"]:
                confirm = input("Are you sure to proceed? [yes/no]: ")
            if confirm == "yes":
                confirmRequest = { "requestType": "STORE_CON", "userName": username }
                confirmResponse = self.call(confirmRequest)
                if "responseType" not in confirmResponse:
                    sys.exit("\nFATAL: Malformed response, store confirmation expected.\n")
                elif confirmResponse["responseType"] == "STORE_RUN":
                    jobId = confirmResponse["jobId"]
                    print(f"\nJobID: {jobId}")
                    print("Store process started successfully!\n")
                else:
                    sys.exit("FATAL: Unknown response type.\n")
            else:
                sys.exit("\nStore process aborted gracefully.\n")
        elif storeResponse["responseType"] == "ERROR":
            errorCode = storeResponse["errorCode"]
            errorMsg = storeResponse["errorMsg"]
            sys.exit(f"Error code: {errorCode}, Error message: {errorMsg}\n")
        else:
            sys.exit("\nFATAL: Unknown response type.\n")


# Create new AMQPClient object
dataArchiverCli = AMQPClient()

# Check the number of input args
if len(sys.argv) == 3:
    script, cmd, username = sys.argv
else:
    dataArchiverCli.help()

# Check the command passed by the user
if cmd == "cstore" or cmd == "hstore":
    dataArchiverCli.store(cmd, username)
else:
    dataArchiverCli.help()