# A FIFO queue based on a Redis list
#
#

import json
import redis

from config import Config
from job import Job

class JobQueue(object):

    def __init__(self, queueName):
        config = Config("vos_ts.conf")
        self.params = config.loadSection("job_cache")
        self.redisCli = redis.StrictRedis(host = self.params["host"],
                                          port = self.params["port"],
                                          db = self.params["db_sched"])
        self.queueName = queueName

    # Returns the number of jobs in the current queue
    def len(self):
        return self.redisCli.llen(self.queueName)

    # Returns the name of the current queue
    def name(self):
        return self.queueName

    # Gets a copy of the first job without moving it out from the current 
    # queue
    def getJob(self):
        jobObj = self.redisCli.lrange(self.queueName, self.len() - 1, self.len() - 1)[0].decode("utf-8")
        return json.loads(jobObj)

    # Pushes a new job into the queue
    def insertJob(self, jobObj):
        data = { "jobId": jobObj.jobId,
                 "phase": jobObj.phase,
                 "quote": jobObj.quote,
                 "startTime": jobObj.startTime,
                 "endTime": jobObj.endTime,
                 "executionDuration": jobObj.executionDuration,
                 "destruction": jobObj.destruction,
                 "parameters": jobObj.parameters,
                 "jobInfo": jobObj.jobInfo }
        self.redisCli.lpush(self.queueName, json.dumps(data))

    # Moves out a job from the end of the current queue and places it 
    # at the beginning of the next queue (this operation is atomic)
    def moveJobTo(self, nextQueueName):
        self.redisCli.brpoplpush(self.queueName, nextQueueName, timeout = 0)


# Test
#p = JobQueue("pending")
#r = JobQueue("ready")
#j1 = Job()
#j1.setPhase("PENDING")
#j2 = Job()
#j2.setPhase("PENDING")
#p.insertJob(j1)
#p.insertJob(j2)
#print(p.getJob())
#p.moveJobTo(r.name())
#print(r.getJob())
