# TODO:
# - interface with the file catalog (insert)
# - data and permissions cleanup
#

import json
import os
import shutil
import sys

from datetime import datetime as dt

from checksum import Checksum
from file_grouper import FileGrouper
from db_connector import DbConnector
from node import Node
from preprocessor import Preprocessor
from config import Config


class StorePreprocessor(Preprocessor):

    def __init__(self):          
        self.md5calc = Checksum()
        config = Config("vos_ts.conf")
        self.params = config.loadSection("file_grouper")        
        self.fileGrouper = FileGrouper(self.params.getint("min_num_files"), 
                                       self.params.getint("max_dir_size") * (2 ** 30))
        self.params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(self.params["user"], 
                                  self.params["password"], 
                                  self.params["host"], 
                                  self.params.getint("port"), 
                                  self.params["db"])
        self.params = config.loadSection("transfer_node")
        self.storageBasePath = self.params["base_path"]        
        self.jobObj = None
        self.username = None
        self.userId = None
        super(StorePreprocessor, self).__init__()

    # Scan is performed only on the first level!
    def scan(self):
        dirList = []
        fileList = []
        elementList = os.listdir(self.path)
        for el in elementList:
            elPath = self.path + '/' + el
            if os.path.isdir(elPath):
                dirList.append(el)
            elif os.path.isfile(elPath):
                fileList.append(el)
            else:
                sys.exit("FATAL: invalid file/dir.")
        return [ dirList, fileList ]
      
    def scanRecursive(self):
        dirList = []
        fileList = []
        for folder, subfolders, files in os.walk(self.path, topdown = True):
            cwd = os.path.basename(folder)
            if folder != self.path:
                parent = os.path.dirname(folder)
                dirList.append(parent + '/' + cwd)
                i = 0
                for f in files:
                    files[i] = parent + '/' + cwd + '/' + f
                    i += 1
                fileList.append(files)
        return [ dirList, fileList ]

    def prepare(self, username):
        self.username = username
        self.path = "/home/" + username + "/store"
        for folder, subfolders, files in os.walk(self.path):
            os.chown(folder, 0, 0)
            os.chmod(folder, 0o555)
            for s in subfolders:
                os.chown(os.path.join(folder, s), 0, 0)
                os.chmod(os.path.join(folder, s), 0o555)
            for f in files:
                os.chown(os.path.join(folder, f), 0, 0)
                os.chmod(os.path.join(folder, f), 0o555)

    def execute(self):
        # First scan to find crowded dirs
        [ dirs, files ] = self.scan()        
        
        # Create a .tar for all dirs matching the constraints, if any
        for dir in dirs:
            self.fileGrouper.recursive(self.path + '/' + dir)
        
        # Second scan after file grouper execution
        [ dirs, files ] = self.scan()
        timestamp = dt.now().strftime("%Y_%m_%d-%H_%M_%S")
        
        # Case 1: /home/user/store contains both files and dirs
        if files and dirs:
            destPath = self.path + '/' + timestamp
            try:
                os.mkdir(destPath)
            except OSError as error:
                sys.exit(f"FATAL: {error}")
            for file in files:
                srcPath = self.path + '/' + file
                shutil.move(srcPath, destPath)
            for dir in dirs:
                srcPath = self.path + '/' + dir                
                shutil.move(srcPath, destPath)
            self.md5calc.recursive(destPath)
        # Case 2: /home/user/store contains only files    
        elif files and not dirs:
            destPath = self.path + '/' + timestamp
            try:
                os.mkdir(destPath)
            except OSError as error:
                sys.exit(f"FATAL: {error}")
            for file in files:
                srcPath = self.path + '/' + file
                shutil.move(srcPath, destPath)
            self.md5calc.recursive(destPath)
        # Case 3: /home/user/store contains only dirs
        elif not files and dirs:
            for dir in dirs:
                destPath = self.path + '/' + dir
                self.md5calc.recursive(destPath)
        # Case 4: /home/user/store is empty
        else:
            sys.exit("The 'store' directory is empty.")
            
        # Third scan after directory structure 'check & repair'
        [ dirs, files ] = self.scanRecursive()
        
        # File catalog update
        out = open("store_preprocessor_log.txt", "a")
        self.dbConn.connect()
        self.userId = self.dbConn.getRapId(self.username)
        out.write(f"USER: {self.username}\n")
        out.write(f"USER_ID: {self.userId}\n")
        pathPrefix = self.storageBasePath.replace("{username}", self.username)
        for dir in dirs:
            out.write(f"DIR: {dir}\n")            
            out.write(f"pathPrefix: {pathPrefix}\n")
            basePath = os.path.dirname(dir).replace(pathPrefix, "/" + self.username)
            out.write(f"basePath: {basePath}\n")
            cnode = Node(os.path.basename(dir), "container");            
            cnode.setParentPath(basePath)
            cnode.setOwnerID(self.userId)
            cnode.setCreatorID(self.userId)
            self.dbConn.insertNode(cnode, cnode.parentPath)
            
        for flist in files:
            for file in flist:
                out.write(f"FILE: {file}\n")
                if self.md5calc.fileIsValid(file):                    
                    out.write(f"pathPrefix: {pathPrefix}\n")
                    basePath = os.path.dirname(file).replace(pathPrefix, "/" + self.username)
                    out.write(f"basePath: {basePath}\n")
                    dnode = Node(os.path.basename(file), "data")
                    dnode.setParentPath(basePath)
                    dnode.setOwnerID(self.userId)
                    dnode.setCreatorID(self.userId)
                    dnode.setContentMD5(self.md5calc.getMD5(file))                
                    self.dbConn.insertNode(dnode, dnode.parentPath)

        out.write("\n")
        out.close()
        self.dbConn.disconnect()
        
    def run(self):
        while True:
            self.wait()
            if(self.readyQueue.len() <= self.maxReadyJobs and self.pendingQueue.len() > 0):
                self.jobObj = self.pendingQueue.getJob()
                self.username = self.jobObj["jobInfo"]["userName"]
                self.prepare(self.username)
                self.execute()
                self.pendingQueue.moveJobTo("ready")
                print("Job MOVED!")

# Test
#sp = StorePreprocessor()
#sp.prepare("curban")
#sp.execute()