#!/usr/bin/env python
#
# This file is part of vospace-transfer-service
# Copyright (C) 2021 Istituto Nazionale di Astrofisica
# SPDX-License-Identifier: GPL-3.0-or-later
#

import datetime
import logging
import os

from config import Config
from datetime import datetime as dt
from db_connector import DbConnector
from mailer import Mailer
from node import Node
from redis_log_handler import RedisLogHandler
from system_utils import SystemUtils
from tabulate import tabulate
from task_executor import TaskExecutor


class GroupRwExecutor(TaskExecutor):

    def __init__(self):
        self.type = "group_rw_executor"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("mail")
        self.adminEmail = params["admin_email"]
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.resDir = params["res_dir"]
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  1,
                                  self.logger)
        self.systemUtils = SystemUtils()
        self.jobObj = None
        self.jobId = None
        self.requestType = None
        self.vospacePath = None
        self.groupName = None
        self.realGroupName = None
        super(GroupRwExecutor, self).__init__()

    def execute(self):
        """This method adds/removes groups to/from 'group_read' and 'group_write'."""
        try:
            self.logger.info("++++++++++ Start of execution phase ++++++++++")
            self.jobObj.setPhase("EXECUTING")
            self.jobObj.setStartTime(datetime.datetime.now().isoformat())
            self.dbConn.insertJob(self.jobObj)
        
            if self.requestType == "GRPR_ADD":
                self.logger.info(f"User command: vos_group read add {self.groupName[0]} {self.vospacePath}")                
                self.dbConn.updateGroupRead(self.groupName, [], self.vospacePath)
            elif self.requestType == "GRPR_DEL":
                self.logger.info(f"User command: vos_group read del {self.groupName[0]} {self.vospacePath}")
                self.dbConn.updateGroupRead([], self.groupName, self.vospacePath)
            elif self.requestType == "GRPW_ADD":
                self.logger.info(f"User command: vos_group write add {self.groupName[0]} {self.vospacePath}")
                self.dbConn.updateGroupWrite(self.groupName, [], self.vospacePath)
            elif self.requestType == "GRPW_DEL":
                self.logger.info(f"User command: vos_group write del {self.groupName[0]} {self.vospacePath}")
                self.dbConn.updateGroupWrite([], self.groupName, self.vospacePath)
        except Exception:
            self.logger.exception("FATAL: something went wrong during the execution phase.")
            return False
        else:
            self.logger.info("++++++++++ End of execution phase ++++++++++")
            return True
        
    def update(self, status):
        try:
            results = [{"target": ""}]
            self.dbConn.setResults(self.jobId, results)

            m = Mailer(self.logger)
            m.addRecipient(self.adminEmail)
            userEmail = self.dbConn.getUserEmail(self.userId)
            if userEmail != self.adminEmail:
                m.addRecipient(userEmail)
        
            if status == "OK":
                self.jobObj.setPhase("COMPLETED")
                self.jobObj.setEndTime(datetime.datetime.now().isoformat())
                self.dbConn.insertJob(self.jobObj)
                self.logger.info("Job phase updated to COMPLETED.")
        
                if self.requestType == "GRPR_ADD":
                    msg = f"""
Added '{self.realGroupName}' to 'group_read' for {self.vospacePath} and any child nodes.
            """
                elif self.requestType == "GRPR_DEL":
                    msg = f"""
Removed '{self.realGroupName}' from 'group_read' for {self.vospacePath} and any child nodes.
            """
                elif self.requestType == "GRPW_ADD":
                    msg = f"""
Added '{self.realGroupName}' to 'group_write' for {self.vospacePath} and any child nodes.
            """
                elif self.requestType == "GRPW_DEL":
                    msg = f"""
Removed '{self.realGroupName}' from 'group_write' for {self.vospacePath} and any child nodes.
            """
            else:
                msg = "FATAL: something went wrong during the execution phase."
                self.jobObj.setPhase("ERROR")
                self.jobObj.setErrorType("fatal")
                self.jobObj.setErrorMessage(msg)
                self.jobObj.setEndTime(datetime.datetime.now().isoformat())
                self.dbConn.insertJob(self.jobObj)
                self.logger.info("Job phase updated to ERROR.")
            # Send e-mail notification
            m.setMessage("VOSpace group_rw notification", msg)
            m.send()
        except Exception:
            self.logger.exception(f"FATAL: unable to update the database, job ID: {self.jobId}")

    def run(self):
        self.logger.info("Starting group_rw executor...")
        self.setSourceQueueName("group_rw_ready")
        self.setDestinationQueueName("group_rw_terminated")
        while True:
            self.wait()
            try:
                srcQueueLen = self.srcQueue.len()
                destQueueLen = self.destQueue.len()
            except Exception:
                self.logger.exception("Cache error: failed to retrieve queue length.")
            else:
                if srcQueueLen > 0:
                    self.jobObj = self.srcQueue.getJob()
                    self.jobId = self.jobObj.jobId
                    self.userId = self.jobObj.ownerId
                    self.requestType = self.jobObj.jobInfo["requestType"]
                    self.vospacePath = self.jobObj.jobInfo["vospacePath"]
                    self.groupName = [ self.jobObj.jobInfo["groupName"] ]
                    self.realGroupName = self.groupName[0].split("people.")[-1].replace("\\", "")
                    if self.execute():
                        self.update("OK")
                    else:
                        self.update("ERROR")
                    try:
                        if destQueueLen >= self.maxTerminatedJobs:
                            self.destQueue.extractJob()
                        self.destQueue.insertJob(self.jobObj)
                        self.srcQueue.extractJob()
                    except Exception:
                        self.logger.exception(f"Failed to move job {self.jobObj.jobId} from '{self.srcQueue.name()}' to '{self.destQueue.name()}'")
                    else:
                        self.logger.info(f"Job {self.jobObj.jobId} MOVED from {self.srcQueue.name()} to {self.destQueue.name()}")
