#!/usr/bin/env python
#
# This file is part of vospace-transfer-service
# Copyright (C) 2021 Istituto Nazionale di Astrofisica
# SPDX-License-Identifier: GPL-3.0-or-later
#

import json
import logging
import os
import paramiko
import scp
import sys

from config import Config
from exceptions import ScpInvalidFileException
from exceptions import TapeClientException
from redis_log_handler import RedisLogHandler
from system_utils import SystemUtils


class TapeClient(object):
    # 'dsmdf' command location on the tape library frontend
    DSMDF = "/usr/bin/dsmdf"

    # 'dsmrecall' command location on the tape library frontend
    DSMRECALL = "/usr/bin/dsmrecall"

    # 'dsmmigrate' command location on the tape library frontend
    DSMMIGRATE = "/usr/bin/dsmmigrate"

    # destination for the files containing the lists of files to recall or migrate
    VOSPACE_WD = "/tmp/vospace"

    def __init__(self, host, port, user, keyFile, logger):
        self.host = host
        self.port = port
        self.user = user
        self.logger = logger
        self.client = paramiko.SSHClient()
        self.client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        self.keyFile = keyFile
        self.scp = None
        self.systemUtils = SystemUtils()
        self.HSMFilesystemList = []

    def connect(self):
        """Connects to the tape library frontend."""
        try:
            self.key = paramiko.RSAKey.from_private_key_file(self.keyFile)
            self.client.load_system_host_keys()
            self.client.connect(hostname = self.host,
                                port = self.port,
                                username = self.user,
                                pkey = self.key)
        except Exception:
            self.logger.exception("Unable to establish SSH connection with tape library frontend.")
            raise

    def copy(self, srcPath, destPath):
        """Copies files/dirs recursively by passing their absolute paths."""
        try:
            self.scp = scp.SCPClient(self.client.get_transport())
        except Exception:
            self.logger.error("Unable to get transport from SSH client.")
            raise
        else:
            self.logger.debug(f"Copying {srcPath} in {destPath}")
            if os.path.isdir(srcPath):
                self.scp.put(srcPath, recursive = True, remote_path = destPath)
            elif os.path.isfile(srcPath):
                self.scp.put(srcPath, destPath)
            else:
                self.logger.error("FATAL: invalid file/dir.")
                raise ScpInvalidFileException
        finally:
            self.scp.close()

    def getHSMFilesystemList(self):
        """Returns a list containing all the available HSM filesystems."""
        cmd = f"{self.DSMDF} -detail | grep \"HSM Filesystem\" | awk '{{ print $3 }}'"
        try:
            stdin, stdout, stderr = self.client.exec_command(cmd)
        except Exception:
            self.logger.exception(f"Unable to execute command: '{cmd}'")
            raise
        else:
            exitCode = stdout.channel.recv_exit_status()
            if not exitCode:
                self.HSMFilesystemList = [ el.rstrip('\n') for el in stdout.readlines() ]
                return self.HSMFilesystemList.copy()
            else:
                raise TapeClientException(cmd, exitCode, stderr)

    def getHSMFilesystemFreeSpace(self, HSMFilesystem):
        """Returns the free space in bytes for a given HSM filesystem."""
        cmd = f"{self.DSMDF} -detail {HSMFilesystem} | grep \"Free Size\" | awk '{{ print $3 }}'"
        try:
            stdin, stdout, stderr = self.client.exec_command(cmd)
        except Exception:
            self.logger.exception(f"Unable to execute command: '{cmd}'")
            raise
        else:
            exitCode = stdout.channel.recv_exit_status()
            if not exitCode:
                result = stdout.readlines()[0].rstrip('\n') + " KB"
                freeSpace = self.systemUtils.convertSizeToBytes(result)
                return freeSpace
            else:
                raise TapeClientException(cmd, exitCode, stderr)

    def migrate(self, fileList, tapeHSMFilesystem, jobId):
        """
        Migrates to tape all files whose absolute path is
        contained in 'fileList'.
        A HSM filesystem and a VOSpace jobId are also
        required as parameters.
        """
        self.logger.info(f"Starting MIGRATE operation (tape HSM filesystem = '{tapeHSMFilesystem}')...")
        migrateFileList = f"vos_migrate-{jobId}.lst"
        try:
            fp = open(migrateFileList, "a")
        except IOError:
            raise
        else:
            for f in fileList:
                fp.write(f"{f}\n")
            fp.close()
            self.copy(f"./{migrateFileList}", f"{self.VOSPACE_WD}/{migrateFileList}")
            os.remove(f"./{migrateFileList}")
            cmd = f"{self.DSMMIGRATE} -filelist={self.VOSPACE_WD}/{migrateFileList} > /dev/null 2>&1"
            try:
                stdin, stdout, stderr = self.client.exec_command(cmd)
            except Exception:
                self.logger.exception(f"Unable to execute command: '{cmd}'")
                raise
            else:
                exitCode = stdout.channel.recv_exit_status()
                if exitCode == 0 or exitCode == 4:
                    self.logger.info("MIGRATE operation COMPLETED.")
                else:
                    self.logger.error("MIGRATE operation FAILED.")
                    raise TapeClientException(cmd, exitCode, stderr)
                return exitCode

    def recall(self, fileList, jobId):
        """
        Recalls from tape all files whose absolute path is
        contained in 'fileList'.
        A VOSpace job ID is also required as parameter.
        """
        self.logger.info("Starting RECALL operation...")
        recallFileList = f"vos_recall-{jobId}.lst"
        try:
             fp = open(recallFileList, "a")
        except IOError:
            raise
        else:
            for f in fileList:
                fp.write(f"{f}\n")
            fp.close()
            self.copy(f"./{recallFileList}", f"{self.VOSPACE_WD}/{recallFileList}")
            os.remove(f"./{recallFileList}")
            cmd = f"{self.DSMRECALL} -filelist={self.VOSPACE_WD}/{recallFileList} > /dev/null 2>&1"
            try:
                stdin, stdout, stderr = self.client.exec_command(cmd)
            except Exception:
                self.logger.exception(f"Unable to execute command: '{cmd}'")
                raise
            else:
                exitCode = stdout.channel.recv_exit_status()
                if exitCode == 0 or exitCode == 4:
                    self.logger.info("RECALL operation COMPLETED.")
                else:
                    self.logger.error("RECALL operation FAILED.")
                    raise TapeClientException(cmd, exitCode, stderr)
                return exitCode

    def recallChecksumFiles(self, dirName):
        """
        Recursively recalls from tape all the checksum files related to
        the 'dirName' directory.
        """
        self.logger.info("Starting RECALL_CHECKSUM operation...")
        checksumFileList = "vos_recall_checksum_files-{jobId}.lst"
        cmd = f"find $(dirname {dirName}) -type f \( -iname \"*-md5sum.txt\" \) > {self.VOSPACE_WD}/{checksumFileList} && {self.DSMRECALL} -filelist={self.VOSPACE_WD}/{checksumFileList} > /dev/null 2>&1"
        try:
            stdin, stdout, stderr = self.client.exec_command(cmd)
        except Exception:
            self.logger.exception(f"Unable to execute command: '{cmd}'")
            raise
        else:
            exitCode = stdout.channel.recv_exit_status()
            if exitCode == 0 or exitCode == 4:
                self.logger.info("RECALL_CHECKSUM operation COMPLETED.")
            else:
                self.logger.error("RECALL_CHECKSUM operation FAILED.")
                raise TapeClientException(cmd, exitCode, stderr)
            return exitCode

    def disconnect(self):
        """Performs a cleanup and closes the connection."""
        self.HSMFilesystemList.clear()
        self.client.close()

    def getSize(self, fsMountPoint):
        """
        DEPRECATED
        """
        cmd = f"df {fsMountPoint} | tail -n +2"
        #out = subprocess.run(cmd, shell = True, capture_output = True)
        stdin, stdout, stderr = self.client.exec_command(cmd)
        #res = stdout.stdout.decode('UTF-8').rstrip('\n')
        res = stdout.readlines()[0]
        res = ' '.join(res.split()).split(' ')
        #print(res)
        total = res[1]
        used = res[2]
        available = res[3]
        return ( total, used, available)


# Test
#tc = TapeClient("192.168.56.101", 22, "root", "ibm")
#tc.connect()
#tc.copy("/home/curban/store/mydir", "/home/mydir")
#tc.copy("/home/curban/store/foo2.txt", "/home/mydir/foo2.txt")
#fsSize = tc.getSize("/ia2_tape_stb_01")
#print(fsSize)
#tc.disconnect()
#for i in tl:
#    print(i.id)
#    print('\n')
