import os
import shutil
import subprocess
import sys

from config import Config
from db_connector import DbConnector
from system_utils import SystemUtils
from tape_client import TapeClient
from task_executor import TaskExecutor


class StoreExecutor(TaskExecutor):

    def __init__(self):
        config = Config("vos_ts.conf")
        self.params = config.loadSection("spectrum_archive")
        self.tapeClient = TapeClient(self.params["host"],
                                     self.params.getint("port"),
                                     self.params["user"],
                                     self.params["password"])
        self.params = config.loadSection("transfer_node")
        self.storageStorePath = self.params["store_path"]
        self.params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(self.params["user"],
                                  self.params["password"],
                                  self.params["host"],
                                  self.params.getint("port"),
                                  self.params["db"])
        self.jobObj = None
        self.jobId = None
        self.username = None
        self.requestType = None
        self.storageId = None
        self.nodeList = []
        self.systemUtils = SystemUtils()
        super(StoreExecutor, self).__init__()

    def copyData(self):
        self.dbConn.connect()
        self.dbConn.setPhase(self.jobId, "EXECUTING")
        srcPathPrefix = self.storageStorePath.replace("{username}", self.username)
        srcData = os.listdir(srcPathPrefix)
        destPathPrefix = self.dbConn.getStorageBasePath(self.storageId) + '/' + self.username
        self.dbConn.disconnect()
        sp = subprocess.run(["rsync", "-av", srcPathPrefix + '/', destPathPrefix + '/'], capture_output = True)
        if(sp.returncode or sp.stderr):
            return False
        else:
            return True

    def cleanup(self):
        srcPathPrefix = self.storageStorePath.replace("{username}", self.username)
        srcData = os.listdir(srcPathPrefix)
        for el in srcData:
            nodeOSPath = srcPathPrefix + '/' + el
            if os.path.isdir(nodeOSPath):
                shutil.rmtree(nodeOSPath)
            elif os.path.isfile(nodeOSPath):
                os.remove(nodeOSPath)
            else:
                sys.exit("Unable to remove file/dir on the transfer node!!!")
        userInfo = self.systemUtils.userInfo(self.username)
        uid = userInfo[1]
        gid = userInfo[2]
        os.chown(srcPathPrefix, uid, gid)
        os.chmod(srcPathPrefix, 0o755)

    def update(self):
        self.dbConn.connect()
        out = open("store_executor_log.txt", "a")
        results = [{"target": ""}]                
        self.dbConn.setResults(self.jobId, results)
        for nodeVOSPath in self.nodeList:
            out.write(f"nodeListElement: {nodeVOSPath}\n")
            self.dbConn.setAsyncTrans(nodeVOSPath, True);
            self.dbConn.setSticky(nodeVOSPath, True);
            self.dbConn.setBusyState(nodeVOSPath, False);
        self.jobObj.setPhase("COMPLETED")
        self.dbConn.setPhase(self.jobId, "COMPLETED")
        out.close()
        self.dbConn.disconnect()

    def run(self):
        print("Starting store executor...")
        self.setSourceQueueName("write_ready")
        self.setDestinationQueueName("write_terminated")
        while True:
            self.wait()
            if self.srcQueue.len() > 0:
                self.jobObj = self.srcQueue.getJob()
                self.jobId = self.jobObj.jobId
                self.username = self.jobObj.jobInfo["userName"]
                self.requestType = self.jobObj.jobInfo["requestType"]
                self.storageId = self.jobObj.jobInfo["storageId"]
                self.nodeList = self.jobObj.jobInfo["nodeList"]
                # 1) Controlla il tipo di destinazione: hot (server) o cold (tape)
                # *) HOT
                # 1) Controlla di avere sufficiente spazio su disco sul server
                # 2) Ottieni la cartella o la lista delle cartelle sulla 'store' sul transf. node
                # 3) Avvia la copia delle cartelle con rsync da transf. node a frontend tape
                # *) COLD
                # 1) Controlla che altri non stiano migrando in questo preciso momento (con tape client)
                # 2) Ottieni la cartella o la lista delle cartelle sulla 'store' sul transf. node
                # 3) Avvia la copia delle cartelle con rsync da transf. node a frontend tape
                # 4) A copia finita, se tutto ok, rimuovi i dati sul tn e setta async_trans a true sul db
                self.copyData()
                self.cleanup()
                self.update()
                self.destQueue.insertJob(self.jobObj)
                self.srcQueue.extractJob()
                print(f"Job {self.jobObj.jobId} MOVED from {self.srcQueue.name()} to {self.destQueue.name()}")
