#!/usr/bin/env python

import json
import logging
import os
import sys

from rap_client import RapClient
from redis_log_handler import RedisLogHandler
from redis_rpc_server import RedisRPCServer
from config import Config
from db_connector import DbConnector
from job import Job
from job_queue import JobQueue
from system_utils import SystemUtils


class DataRPCServer(RedisRPCServer):

    def __init__(self, host, port, db, rpcQueue):
        self.type = "data"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  2)
        params = config.loadSection("transfer_node")
        self.storageStorePath = params["store_path"]
        params = config.loadSection("scheduling")
        self.maxPendingJobs = params.getint("max_pending_jobs")
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        self.pendingQueueWrite = JobQueue("write_pending")
        self.systemUtils = SystemUtils()
        super(DataRPCServer, self).__init__(host, port, db, rpcQueue)

    def callback(self, requestBody):
        try:
            queueLen = self.pendingQueueWrite.len()
        except Exception:
            errorMsg = "Cache error."
            self.logger.exception(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 3,
                         "errorMsg": errorMsg }
            return response
        # 'requestType' and 'userName' attributes are mandatory
        if "requestType" not in requestBody or "userName" not in requestBody:
            errorMsg = "Malformed request, missing parameters."
            self.logger.error(errorMsg)
            response = { "responseType":"ERROR",
                         "errorCode": 1,
                         "errorMsg": errorMsg }
        elif queueLen >= self.maxPendingJobs:
            errorMsg = "Pending queue is full, please, retry later."
            self.logger.error(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 4,
                         "errorMsg":  errorMsg }
        elif requestBody["requestType"] == "CSTORE" or requestBody["requestType"] == "HSTORE":
            user = requestBody["userName"]
            try:
                userInDb = self.dbConn.userExists(user)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response            
            try:
                rapCli = RapClient()
                rapInfo = rapCli.getUserInfo(user)
            except MultipleUsersException:
                errorMsg = "Multiple users with the same email address in RAP."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 5,
                             "errorMsg": errorMsg }
                return response
            except HTTPException:
                errorMsg = "HTTP exception."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 6,
                             "errorMsg": errorMsg }
                return response                
            try:    
                if requestBody["requestType"] == "CSTORE":
                    storageList = self.dbConn.getStorageListByType("cold")
                else:
                    storageList = self.dbConn.getStorageListByType("hot")
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            #folderPath = "/home/" + user + "/store"
            folderPath = self.storageStorePath.replace("{username}", user)
            userInfo = self.systemUtils.userInfo(user)
            # Check if the user exists on the transfer node and is registered in the database
            if not userInfo:
                # the user does not exist on the system
                errorMsg = "The user does not exist on the transfer node."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 7,
                             "errorMsg": errorMsg }
            elif not (userInDb or rapInfo):
                # the user cannot be found neither in RAP nor in db
                errorMsg = "The user is not registered neither in RAP nor in the database."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 8,
                             "errorMsg": errorMsg }
            else:
                if not userInDb and rapInfo:
                    # retrieve data from RAP and insert them in db
                    try:
                        self.dbConn.insertUser(rapInfo["id"], user, rapInfo["email"])
                    except Exception:
                        errorMsg = "Database error."
                        self.logger.exception(errorMsg)
                        response = { "responseType": "ERROR",
                                     "errorCode": 2,
                                     "errorMsg": errorMsg }
                        return response
                uid = os.stat(folderPath).st_uid
                gid = os.stat(folderPath).st_gid
                # Check if uid and gid match and avoid privilege escalation
                if uid == userInfo[1] and gid == userInfo[2] and uid >= 1000 and gid >= 100:
                    # If write permissions are set and the 'store' folder is not empty,
                    # it means that data is ready to be copied, otherwise, nothing can
                    # be done until the write permissions are restored or new data is
                    # copied on the transfer node by the user.
                    if os.access(folderPath, os.W_OK) and os.listdir(folderPath):
                        response = { "responseType": "STORE_ACK",
                                     "storageList": storageList }
                    elif os.access(folderPath, os.W_OK) and not os.listdir(folderPath):
                        errorMsg = "The 'store' directory on the transfer node is empty."
                        self.logger.error(errorMsg)
                        response = { "responseType": "ERROR",
                                     "errorCode": 9,
                                     "errorMsg": errorMsg }
                    else:
                        errorMsg = "Service busy. Please, retry later."
                        self.logger.error(errorMsg)
                        response = { "responseType": "ERROR",
                                     "errorCode": 10,
                                     "errorMsg": errorMsg }
                else:
                    errorMsg = "Permission denied."
                    self.logger.error(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 11,
                                 "errorMsg": errorMsg }
        elif requestBody["requestType"] == "STORE_CON":
            username = requestBody["userName"]
            self.prepare(username)
            jobObj = Job()
            jobObj.setType("vos_data")
            jobObj.setInfo(requestBody)
            jobObj.setPhase("PENDING")
            try:
                jobObj.setOwnerId(self.dbConn.getUserId(username))
                self.dbConn.insertJob(jobObj)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            jobObj.jobInfo["storageId"] = requestBody["storageId"]
            jobObj.jobInfo["storageType"] = requestBody["storageType"]
            try:
                self.pendingQueueWrite.insertJob(jobObj)
            except Exception:
                errorMsg = "Unable to insert the job in the 'write_pending' queue."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 12,
                             "errorMsg": errorMsg }
                return response
            else:
                response = { "responseType": "STORE_RUN", 
                             "jobId": jobObj.jobId }
        else:
            errorMsg = "Unkown request type."
            self.logger.error(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 13,
                         "errorMsg": errorMsg }
        return response

    def prepare(self, username):
        #path = "/home/" + username + "/store"
        path = self.storageStorePath.replace("{username}", username)
        for el in os.listdir(path):
            absPath = path + '/' + el
            os.chown(absPath, 0, 0)
            os.chmod(absPath, 0o444)

    def run(self):
        self.logger.info(f"Starting RPC server of type {self.type}...")
        super(DataRPCServer, self).run()
