#!/usr/bin/env python

import json
import logging
import os

from config import Config
from db_connector import DbConnector
from mailer import Mailer
from redis_log_handler import RedisLogHandler
from task_executor import TaskExecutor


class RetrievePreprocessor(TaskExecutor):

    def __init__(self):
        self.type = "retrieve_preprocessor"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  1)
        params = config.loadSection("mail")
        self.adminEmail = params["admin_email"]
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        self.jobObj = None
        self.nodeList = []
        super(RetrievePreprocessor, self).__init__()

    def execute(self):
        self.logger.info("Generating VOSpace node list")
        target = self.jobObj.jobInfo["transfer"]["target"].split("!vospace")[1]
        params = self.jobObj.jobInfo["transfer"]["view"]["param"]      
        if not params:
            self.nodeList.append(target)
        else:
            for el in params:
                self.nodeList.append(target + '/' + el["value"])
        self.jobObj.jobInfo["nodeList"] = self.nodeList.copy()
        
    def update(self):
        # Send e-mail notification
        m = Mailer(self.logger)
                
        m.addRecipient(self.adminEmail)
        userEmail = self.dbConn.getUserEmail(self.jobObj.ownerId)
        if userEmail != self.adminEmail:
            m.addRecipient(userEmail)
        
        msg = f"""
        Dear user,
        your job has been QUEUED.

        Job ID: {self.jobObj.jobId}
        Job type: {self.jobObj.type}
        Owner ID: {self.jobObj.ownerId}

        You will be notified by email once the job is completed.

        """
        m.setMessage("VOSpace data retrieve notification: Job QUEUED", msg)
        m.send()

    def cleanup(self):
        self.nodeList.clear()

    def run(self):
        self.logger.info("Starting retrieve preprocessor...")
        self.setSourceQueueName("read_pending")
        self.setDestinationQueueName("read_ready")
        while True:
            self.wait()
            try:
                srcQueueLen = self.srcQueue.len()
                destQueueLen = self.destQueue.len()
            except:
                self.logger.exception("Cache error: failed to retrieve queue length.")
            else:
                if destQueueLen < self.maxReadyJobs and srcQueueLen > 0:
                    self.jobObj = self.srcQueue.getJob()
                    self.execute()
                    self.update()
                    try:
                        self.destQueue.insertJob(self.jobObj)
                        self.srcQueue.extractJob()
                    except Exception:
                        self.logger.exception(f"Failed to move job {self.jobObj.jobId} from '{self.srcQueue.name()}' to '{self.destQueue.name()}'")
                    else:                        
                        self.logger.info(f"Job {self.jobObj.jobId} MOVED from '{self.srcQueue.name()}' to '{self.destQueue.name()}'")
            finally:
                self.cleanup()
