#!/usr/bin/env python

import logging

from config import Config
from cli_handler import CliHandler
from job_scheduler import JobScheduler
from log_listener import LogListener
from redis_log_handler import RedisLogHandler
from vospace_rest_handler import VOSpaceRestHandler


class TransferService(object):

    def __init__(self):
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        logStreamHandler = logging.StreamHandler()
        logStreamHandler.setFormatter(logFormatter)
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.logger.addHandler(logStreamHandler)
        params = config.loadSection("job_cache")
        self.cliHandler = CliHandler(params["host"], params.getint("port"), params.getint("db_sched"))
        self.vosRestHandler = VOSpaceRestHandler(params["host"], params.getint("port"), params.getint("db_sched"))
        self.jobScheduler = JobScheduler()

        # PullToVOSpace (via REST API)
        self.vosRestHandler.addRPCServer("start", "start_job_queue")
        self.vosRestHandler.addRPCServer("poll", "poll_job_queue")
        self.vosRestHandler.addRPCServer("abort", "abort_job_queue")

        # PushToVOSpace (via vos_data, the 'unofficial' command line client)
        self.cliHandler.addRPCServer("data", "data_queue")

        # Group
        self.cliHandler.addRPCServer("group_rw", "group_queue")

        # Import
        self.cliHandler.addRPCServer("import", "import_queue")

        # Job
        self.cliHandler.addRPCServer("job", "job_queue")

        # Storage
        self.cliHandler.addRPCServer("storage", "storage_queue")

        # Job scheduler task executors
        self.jobScheduler.addTaskExecutor("group_rw_executor")
        self.jobScheduler.addTaskExecutor("import_executor")
        self.jobScheduler.addTaskExecutor("retrieve_preprocessor")
        self.jobScheduler.addTaskExecutor("store_preprocessor")
        self.jobScheduler.addTaskExecutor("retrieve_executor")
        self.jobScheduler.addTaskExecutor("store_executor")
        self.jobScheduler.addTaskExecutor("retrieve_cleaner")

        # Log listener
        self.logListener = LogListener()

    def start(self):
        # Startup
        self.logListener.start()
        self.jobScheduler.start()
        self.vosRestHandler.start()
        self.cliHandler.start()
        self.logger.info("========== VOSpace Transfer Service is RUNNING! ==========")

ts = TransferService()
ts.start()
