import os
from amqp_server import AMQPServer
from checksum import Checksum
from db_connector import DbConnector
from node import Node
from system_utils import SystemUtils
from config import Config


class ImportAMQPServer(AMQPServer):

    def __init__(self, host, port, queue):
        self.type = "import"
        self.md5calc = Checksum()
        config = Config("vos_ts.conf")       
        self.params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(self.params["user"], 
                                  self.params["password"], 
                                  self.params["host"], 
                                  self.params.getint("port"), 
                                  self.params["db"])
        self.systemUtils = SystemUtils()
        self.path = None
        self.username = None
        super(ImportAMQPServer, self).__init__(host, port, queue)

    def execute_callback(self, requestBody):
        # 'requestType' and 'path' attributes are mandatory
        if "requestType" not in requestBody or "path" not in requestBody:
            response = { "errorCode": 1, "errorMsg": "Malformed request, missing parameters." }
        elif requestBody["requestType"] == "NODE_IMPORT":
            self.path = os.path.abspath(requestBody["path"])
            self.username = requestBody["userName"]
            self.dbConn.connect()
            userInDb = self.dbConn.userExists(self.username)
            userInfo = self.systemUtils.userInfo(self.username)
            out = open("import_amqp_server_log.txt", "a")
            
            if not userInfo or not userInDb:
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": "The user does not exist or is not registered in the database." }
                self.dbConn.disconnect()
                return response
            
            userId = self.dbConn.getRapId(self.username)
            pathPrefix = self.dbConn.storageBasePathIsValid(self.path)
            
            if pathPrefix:
                storageId = self.dbConn.getStorageId(pathPrefix)            
            else:
                response = { "responseType": "ERROR",
                             "errorCode": 3,
                             "errorMsg": "Invalid storage mount point." }
                self.dbConn.disconnect()
                return response
            
            if not os.path.exists(self.path):
                response = { "responseType": "ERROR",
                             "errorCode": 4,
                             "errorMsg": "Path not found." }
                self.dbConn.disconnect()
                return response
            elif not os.path.isdir(self.path):
                response = { "responseType": "ERROR",
                             "errorCode": 5,
                             "errorMsg": "Directory path expected." }
                self.dbConn.disconnect()
                return response
            elif self.username not in self.path:
                response = { "responseType": "ERROR",
                             "errorCode": 6,
                             "errorMsg": "Directory path does not contain the username." }
                self.dbConn.disconnect()
                return response
            elif os.path.dirname(self.path) != pathPrefix + '/' + self.username:
                response = { "responseType": "ERROR",
                             "errorCode": 7,
                             "errorMsg": "Invalid path, directory must be located in " + pathPrefix + '/' + self.username  }
                self.dbConn.disconnect()
                return response            
            else:             
                [ dirs, files ] = self.systemUtils.scanRecursive(os.path.dirname(self.path))
                
                for dir in dirs:            
                    out.write(f"DIR dir: {dir}\n")
                    out.write(f"DIR pathPrefix: {pathPrefix}\n\n")
                    
                    if self.path in dir:
                        parentPath = os.path.dirname(dir).split(pathPrefix)[1]
                        nodeName = os.path.basename(dir)
                        
                        if parentPath == '/':
                            vospacePath = parentPath + nodeName
                        else:
                            vospacePath = parentPath + '/' + nodeName

                        cnode = Node(nodeName, "container")
                        cnode.setParentPath(parentPath)
                        locationId = self.dbConn.getLocationId(storageId)
                        cnode.setLocationId(locationId)
                        cnode.setOwnerID(userId)
                        cnode.setCreatorID(userId)
                        cnode.setContentLength(0)
                        if not self.dbConn.nodeExists(cnode):
                            self.dbConn.insertNode(cnode)
                            self.dbConn.setAsyncTrans(vospacePath, True)
                            self.dbConn.setSticky(vospacePath, True)

                for flist in files:
                    for file in flist:                
                        if self.md5calc.fileIsValid(file) and self.path in os.path.dirname(file):
                            out.write(f"FILE files: {files}\n")
                            out.write(f"FILE flist: {flist}\n")
                            out.write(f"FILE file: {file}\n")
                            out.write(f"FILE pathPrefix: {pathPrefix}\n")
                            parentPath = os.path.dirname(file).split(pathPrefix)[1]
                            out.write(f"FILE parentPath: {parentPath}\n")
                            nodeName = os.path.basename(file)
                            out.write(f"FILE nodeName: {nodeName}\n")
                            vospacePath = parentPath + '/' + nodeName                        
                            dnode = Node(nodeName, "data")
                            out.write(f"FILE vospacePath: {vospacePath}\n")
                            dnode.setParentPath(parentPath)
                            storageId = self.dbConn.getStorageId(pathPrefix)
                            locationId = self.dbConn.getLocationId(storageId)
                            dnode.setLocationId(locationId)
                            dnode.setOwnerID(userId)
                            dnode.setCreatorID(userId)
                            dnode.setContentLength(os.path.getsize(file))
                            dnode.setContentMD5(self.md5calc.getMD5(file))
                            
                            if not self.dbConn.nodeExists(dnode):
                                self.dbConn.insertNode(dnode)
                                self.dbConn.setAsyncTrans(vospacePath, True)
                                self.dbConn.setSticky(vospacePath, True)

            self.dbConn.disconnect()
            # add a counter to track the number of nodes (files and dirs) + log file
            response = { "responseType": "IMPORT_DONE" }
        else:
            response = { "responseType": "ERROR",
                         "errorCode": 8,
                         "errorMsg": "Unkown request type." }

        return response

    def run(self):
        print(f"Starting AMQP server of type {self.type}...")
        super(ImportAMQPServer, self).run()
 
