#  TODO:
#  - error codes and status codes list and description
#  - check what happens if the user folder does not exist (e.g. /home/ccarbone)
#
#

import os
import sys
import json

#from enum import Enum

from amqp_server import AMQPServer
from db_connector import DbConnector
from system_utils import SystemUtils
from job import Job
from job_queue import JobQueue
from config import Config


class StoreAMQPServer(AMQPServer):

    def __init__(self, host, port, queue):
        self.type = "store"
        self.storeAck = False
        config = Config("vos_ts.conf")       
        self.params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(self.params["user"], 
                                  self.params["password"], 
                                  self.params["host"], 
                                  self.params.getint("port"), 
                                  self.params["db"])
        self.params = config.loadSection("transfer_node")
        self.storageStorePath = self.params["store_path"]
        self.pendingQueueWrite = JobQueue("write_pending")
        self.systemUtils = SystemUtils()
        self.job = None
        self.username = None
        self.path = None
        super(StoreAMQPServer, self).__init__(host, port, queue)

    def execute_callback(self, requestBody):
        # 'requestType' and 'userName' attributes are mandatory
        if "requestType" not in requestBody or "userName" not in requestBody:
            response = { "errorCode": 1, "errorMsg": "Malformed request, missing parameters." }
        elif requestBody["requestType"] == "CSTORE" or requestBody["requestType"] == "HSTORE":
            user = requestBody["userName"]
            self.job = Job()
            self.job.setType("other")
            self.job.setInfo(requestBody)
            self.job.setPhase("PENDING")
            self.dbConn.connect()
            userInDb = self.dbConn.userExists(user)
            if requestBody["requestType"] == "CSTORE":
                storageList = self.dbConn.getStorageListByType("cold")
            else:
                storageList = self.dbConn.getStorageListByType("hot")
            self.dbConn.disconnect()
            #folderPath = "/home/" + user + "/store"
            folderPath = self.storageStorePath.replace("{username}", user)
            userInfo = self.systemUtils.userInfo(user)
            # Check if the user exists on the transfer node and is registered in the database
            if not userInfo or not userInDb:
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": "The user does not exist on the transfer node or is not registered in the database." }
            else:
                uid = os.stat(folderPath).st_uid
                gid = os.stat(folderPath).st_gid
                # Check if uid and gid match and avoid privilege escalation
                if uid == userInfo[1] and gid == userInfo[2] and uid != 0 and gid != 0:
                    # If write permissions are set and the 'store' folder is not empty,
                    # it means that data is ready to be copied, otherwise, nothing can
                    # be done until the write permissions are restored or new data is
                    # copied on the transfer node by the user.
                    if os.access(folderPath, os.W_OK) and os.listdir(folderPath):
                        response = { "responseType": "STORE_ACK",
                                     "storageList": storageList }
                        self.storeAck = True
                    elif os.access(folderPath, os.W_OK) and not os.listdir(folderPath):
                        response = { "responseType": "ERROR",
                                     "errorCode": 3,
                                     "errorMsg": "The 'store' directory on the transfer node is empty." }
                    else:
                        response = { "responseType": "ERROR",
                                     "errorCode": 4,
                                     "errorMsg": "Service busy. Please, retry later." }
                else:
                    response = { "responseType": "ERROR",
                                 "errorCode": 5,
                                 "errorMsg": "Permission denied." }
        elif requestBody["requestType"] == "STORE_CON":
            if self.storeAck:
                self.storeAck = False
                user = requestBody["userName"]
                self.prepare(user)
                self.dbConn.connect()
                self.job.setOwnerId(self.dbConn.getRapId(self.username))
                self.dbConn.insertJob(self.job)
                dbResponse = self.dbConn.getJob(self.job.jobId)
                self.dbConn.disconnect()
                self.job.jobInfo["storageId"] = requestBody["storageId"]
                self.pendingQueueWrite.insertJob(self.job)
                if "error" in dbResponse:
                    response = { "responseType": "ERROR",
                                 "errorCode": 6,
                                 "errorMsg": "Job creation failed." }
                else:
                    response = { "responseType": "STORE_RUN", 
                                 "jobId": self.job.jobId }
            else:
                response = { "responseType": "ERROR",
                             "errorCode": 7,
                             "errorMsg": "Store request not acknowledged." }
        else:
            response = { "responseType": "ERROR",
                         "errorCode": 8,
                         "errorMsg": "Unkown request type." }

        return response

    # to be removed from store_preprocessor.py
    # or simply add a chmod -x here, to be faster?
    def prepare(self, username):
        self.username = username
        #self.path = "/home/" + username + "/store"
        self.path = self.storageStorePath.replace("{username}", self.username)
        for folder, subfolders, files in os.walk(self.path):
            os.chown(folder, 0, 0)
            os.chmod(folder, 0o555)
            for s in subfolders:
                os.chown(os.path.join(folder, s), 0, 0)
                os.chmod(os.path.join(folder, s), 0o555)
            for f in files:
                os.chown(os.path.join(folder, f), 0, 0)
                os.chmod(os.path.join(folder, f), 0o555)

    def run(self):
        print(f"Starting AMQP server of type {self.type}...")
        super(StoreAMQPServer, self).run()
