#!/usr/bin/env python
#
# This file is part of vospace-transfer-service
# Copyright (C) 2021 Istituto Nazionale di Astrofisica
# SPDX-License-Identifier: GPL-3.0-or-later
#

import logging
import os

from config import Config
from db_connector import DbConnector
from node import Node
from redis_log_handler import RedisLogHandler
from redis_rpc_server import RedisRPCServer


class UserRPCServer(RedisRPCServer):

    def __init__(self, host, port, db, rpcQueue):
        self.type = "user"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  2,
                                  self.logger)
        super(UserRPCServer, self).__init__(host, port, db, rpcQueue)

    def callback(self, requestBody):
        # 'requestType' attribute is mandatory
        if "requestType" not in requestBody:
            errorMsg = "Malformed request, missing parameters."
            response = { "responseType": "ERROR",
                         "errorCode": 1,
                         "errorMsg": errorMsg }
        elif requestBody["requestType"] == "USER_ADD":
            userId = requestBody["userId"]
            username = requestBody["username"]
            email = requestBody["email"]
            try:
                result = self.dbConn.insertUser(userId,
                                                username,
                                                email)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            else:
                if result:
                    userNode = Node(username, "container")
                    userNode.setCreatorId(userId)
                    userNode.setParentPath("")
                    userNode.setSticky(True)
                    try:
                        self.dbConn.insertNode(userNode)
                    except Exception:
                        errorMsg = "Database error."
                        self.logger.exception(errorMsg)
                        response = { "responseType": "ERROR",
                                     "errorCode": 2,
                                     "errorMsg": errorMsg }
                        return response
                    else:
                        response = { "responseType": "USER_ADD_DONE" }
                else:
                    errorMsg = "User already exists."
                    self.logger.error(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 3,
                                 "errorMsg": errorMsg }
        elif requestBody["requestType"] == "USER_DEL_REQ":
            try:
                result = self.dbConn.getUserList()
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            else:
                response = { "responseType": "USER_DEL_ACK",
                             "userList": result }
        elif requestBody["requestType"] == "USER_DEL_CON":
            userId = requestBody["userId"]
            try:
                result = self.dbConn.deleteUser(userId)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            else:
                if result:
                    response = { "responseType": "USER_DEL_DONE" }
                else:
                    errorMsg = "The user cannot be removed because its VOSpace node has child nodes."
                    self.logger.error(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 4,
                                 "errorMsg": errorMsg }
        elif requestBody["requestType"] == "USER_SEARCH":
            searchStr = requestBody["searchStr"]
            try:
                result = self.dbConn.searchUsers(searchStr)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            else:
                response = { "responseType": "SEARCH_DONE", 
                             "userSearch": result }
        else:
            errorMsg = "Unkown request type."
            self.logger.error(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 5,
                         "errorMsg": errorMsg }
        return response

    def run(self):
        self.logger.info(f"Starting RPC server of type {self.type}...")
        super(UserRPCServer, self).run()
