#!/usr/bin/env python

import logging
import os

from config import Config
from db_connector import DbConnector
from job_queue import JobQueue
from job import Job
from rap_client import RapClient, MultipleUsersException
from redis_log_handler import RedisLogHandler
from redis_rpc_server import RedisRPCServer
from system_utils import SystemUtils


class ImportRPCServer(RedisRPCServer):

    def __init__(self, host, port, db, rpcQueue):
        self.type = "import"
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("file_catalog")
        self.dbConn = DbConnector(params["user"],
                                  params["password"],
                                  params["host"],
                                  params.getint("port"),
                                  params["db"],
                                  1,
                                  2)
        params = config.loadSection("scheduling")
        self.maxReadyJobs = params.getint("max_ready_jobs")
        params = config.loadSection("logging")
        self.logger = logging.getLogger(__name__)
        logLevel = "logging." + params["log_level"]
        logFormat = params["log_format"]
        logFormatter = logging.Formatter(logFormat)
        self.logger.setLevel(eval(logLevel))
        redisLogHandler = RedisLogHandler()
        redisLogHandler.setFormatter(logFormatter)
        self.logger.addHandler(redisLogHandler)
        self.importReadyQueue = JobQueue("import_ready")
        self.systemUtils = SystemUtils()
        super(ImportRPCServer, self).__init__(host, port, db, rpcQueue)

    def callback(self, requestBody):
        # 'requestType' and 'path' attributes are mandatory
        if "requestType" not in requestBody or "path" not in requestBody:
            errorMsg = "Malformed request, missing parameters."
            self.logger.error(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 1,
                         "errorMsg": errorMsg }
        elif requestBody["requestType"] == "NODE_IMPORT":
            path = os.path.abspath(requestBody["path"])
            username = requestBody["userName"]
            try:
                userInDb = self.dbConn.userExists(username)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            try:
                rapCli = RapClient()
                rapInfo = rapCli.getUserInfo(username)
            except MultipleUsersException:
                errorMsg = "Multiple users with the same email address in RAP."
                response = { "responseType": "ERROR",
                             "errorCode": 4,
                             "errorMsg": errorMsg }
                return response
            except HTTPException:
                errorMsg = "HTTP exception."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 5,
                             "errorMsg": errorMsg }
                return response
            userInfo = self.systemUtils.userInfo(username)
            if not userInfo:
                errorMsg = "The user does not exist on the transfer node."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 6,
                             "errorMsg": errorMsg }
                return response
            elif not (userInDb or rapInfo):
                # the user cannot be found neither in RAP nor in db
                errorMsg = "The user is not registered neither in RAP nor in the database."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 7,
                             "errorMsg": errorMsg }
                return response
            else:
                if not userInDb and rapInfo:
                    # retrieve data from RAP and insert them in db
                    try:
                        self.dbConn.insertUser(rapInfo["id"], username, rapInfo["email"])
                    except Exception:
                        errorMsg = "Database error."
                        self.logger.exception(errorMsg)
                        response = { "responseType": "ERROR",
                                     "errorCode": 2,
                                     "errorMsg": errorMsg }
                        return response
            try:        
                userId = self.dbConn.getUserId(username)
                pathPrefix = self.dbConn.storageBasePathIsValid(path)
            except Exception:
                errorMsg = "Database error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 2,
                             "errorMsg": errorMsg }
                return response
            if pathPrefix:
                storageId = self.dbConn.getStorageId(pathPrefix)
                storageType = self.dbConn.getStorageType(pathPrefix)
            else:
                errorMsg = "Invalid storage mount point."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 8,
                             "errorMsg": errorMsg }
                return response
            try:
                queueLen = self.importReadyQueue.len()
            except Exception:
                errorMsg = "Cache error."
                self.logger.exception(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 3,
                             "errorMsg": errorMsg }
                return response
            if not os.path.exists(path):
                errorMsg = "Path not found."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 9,
                             "errorMsg": errorMsg }
            elif not os.path.isdir(path):
                errorMsg = "Directory path expected."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 10,
                             "errorMsg": errorMsg }
            elif username not in path:
                errorMsg = "Directory path does not contain the username."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 11,
                             "errorMsg": errorMsg }
            elif os.path.dirname(path) != pathPrefix + '/' + username:
                errorMsg = "Invalid path, directory must be located in " + pathPrefix + '/' + username
                response = { "responseType": "ERROR",
                             "errorCode": 12,
                             "errorMsg": errorMsg }                
            elif queueLen >= self.maxReadyJobs:
                errorMsg = "Import queue is full, please, retry later."
                self.logger.error(errorMsg)
                response = { "responseType": "ERROR",
                             "errorCode": 13,
                             "errorMsg": errorMsg }
            else:
                jobObj = Job()
                jobObj.setType("vos_import")
                jobInfo = requestBody.copy()
                jobInfo["pathPrefix"] = pathPrefix
                jobInfo["storageId"] = storageId
                jobInfo["storageType"] = storageType
                jobObj.setInfo(jobInfo)
                jobObj.setPhase("QUEUED")
                jobObj.setOwnerId(userId)
                try:
                    self.dbConn.insertJob(jobObj)
                except Exception:
                    errorMsg = "Database error."
                    self.logger.exception(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 2,
                                 "errorMsg": errorMsg }
                    return response
                try:
                    self.importReadyQueue.insertJob(jobObj)
                except Exception:
                    errorMsg = "Cache error."
                    self.logger.exception(errorMsg)
                    response = { "responseType": "ERROR",
                                 "errorCode": 3,
                                 "errorMsg": errorMsg }
                    return response
                response = { "responseType": "IMPORT_STARTED" }
        else:
            errorMsg = "Unkown request type."
            self.logger.error(errorMsg)
            response = { "responseType": "ERROR",
                         "errorCode": 14,
                         "errorMsg": errorMsg }
        return response

    def run(self):
        self.logger.info(f"Starting RPC server of type {self.type}...")
        super(ImportRPCServer, self).run()
