#!/usr/bin/env python
#
# A FIFO queue based on Redis lists
#
#

import json
import logging
import redis
import time

from redis.exceptions import ConnectionError

from config import Config
from job import Job


class JobQueue(object):

    def __init__(self, queueName):
        config = Config("/etc/vos_ts/vos_ts.conf")
        params = config.loadSection("job_cache")
        self.redisCli = redis.Redis(host = params["host"],
                                    port = params["port"],
                                    db = params["db_sched"])
        self.queueName = queueName

    def len(self, retry = 10, timeout = 30):
        """Returns the number of jobs in the current queue."""
        if retry < 1:
            retry = 1
        if timeout < 1:
            timeout = 1
        while True:
            try:
                numJobs = self.redisCli.llen(self.queueName)
            except ConnectionError:
                if retry > 0:
                    retry -= 1
                    time.sleep(timeout)
                else:
                    raise
            except Exception:
                raise
            else:
                return numJobs

    def name(self):
        """Returns the name of the current queue."""
        return self.queueName

    def getJob(self, retry = 10, timeout = 30):
        """Gets a copy of the first job without moving it out from the current queue."""
        if retry < 1:
            retry = 1
        if timeout < 1:
            timeout = 1        
        while True:
            try:
                job = json.loads(self.redisCli.lrange(self.queueName, -1, -1)[0].decode("utf-8"))
                jobObj = Job()
                jobObj.setId(job["jobId"])
                jobObj.setType(job["jobType"])
                jobObj.setOwnerId(job["ownerId"])
                jobObj.setPhase(job["phase"])
                jobObj.setQuote(job["quote"])
                jobObj.setStartTime(job["startTime"])
                jobObj.setEndTime(job["endTime"])
                jobObj.setExecutionDuration(job["executionDuration"])
                jobObj.setInfo(job["jobInfo"])
                jobObj.setNodeList(job["nodeList"])
            except ConnectionError:
                if retry > 0:
                    retry -= 1
                    time.sleep(timeout)
                else:
                    raise
            except Exception:
                raise
            else:
                return jobObj

    def insertJob(self, jobObj, retry = 10, timeout = 30):
        """Pushes a new job into the queue."""
        if retry < 1:
            retry = 1
        if timeout < 1:
            timeout = 1
        data = { "jobId": jobObj.jobId,
                 "jobType": jobObj.type,
                 "ownerId": jobObj.ownerId,
                 "phase": jobObj.phase,
                 "quote": jobObj.quote,
                 "startTime": jobObj.startTime,
                 "endTime": jobObj.endTime,
                 "executionDuration": jobObj.executionDuration,
                 "destruction": jobObj.destruction,
                 "parameters": jobObj.parameters,
                 "results": jobObj.results,
                 "jobInfo": jobObj.jobInfo, 
                 "nodeList": jobObj.nodeList }
        while True:
            try:
                self.redisCli.lpush(self.queueName, json.dumps(data))
            except ConnectionError:
                if retry > 0:
                    retry -= 1
                    time.sleep(timeout)
                else:
                    raise
            except Exception:
                raise
            else:
                return

    def extractJob(self, retry = 10, timeout = 30):
        """Moves out a job from the end of the current queue."""
        if retry < 1:
            retry = 1
        if timeout < 1:
            timeout = 1
        while True:
            try:
                job = json.loads(self.redisCli.brpop(self.queueName)[1].decode("utf-8"))
            except ConnectionError:
                if retry > 0:
                    retry -= 1
                    time.sleep(timeout)
                else:
                    raise
            except Exception:
                raise
            else:
                jobObj = Job()
                jobObj.setId(job["jobId"])
                jobObj.setType(job["jobType"])
                jobObj.setOwnerId(job["ownerId"])
                jobObj.setPhase(job["phase"])
                jobObj.setQuote(job["quote"])
                jobObj.setStartTime(job["startTime"])
                jobObj.setEndTime(job["endTime"])
                jobObj.setExecutionDuration(job["executionDuration"])
                jobObj.setInfo(job["jobInfo"])
                jobObj.setNodeList(job["nodeList"])
                return jobObj

    def moveJobTo(self, nextQueueName):
        """
        Moves out a job from the end of the current queue and places it 
        at the beginning of the next queue (this operation is atomic)
        DEPRECATED
        """
        try:
            self.redisCli.brpoplpush(self.queueName, nextQueueName)
        except Exception:
            raise


# Test
#p = JobQueue("pending")
#r = JobQueue("ready")
#j1 = Job()
#j1.setPhase("PENDING")
#j2 = Job()
#j2.setPhase("PENDING")
#p.insertJob(j1)
#p.insertJob(j2)
#print(p.getJob())
#p.moveJobTo(r.name())
#print(r.getJob())
